# Backend Starter Kit - Node.js Express API

A professional, production-ready backend starter kit built with Node.js, Express, MySQL, and JWT authentication.

## Table of Contents

- [Features](#features)
- [Tech Stack](#tech-stack)
- [Project Structure](#project-structure)
- [Database Schema](#database-schema)
- [Authentication Flow](#authentication-flow)
- [API Endpoints](#api-endpoints)
- [Installation](#installation)
- [Configuration](#configuration)
- [Usage](#usage)
- [Middleware](#middleware)
- [Security Features](#security-features)
- [Development](#development)
- [Deployment](#deployment)

---

## Features

- ✅ **Email-based Authentication** with verification codes (no passwords)
- ✅ **JWT Token Management** with 10-hour expiration
- ✅ **Master Admin System** with automatic creation
- ✅ **Role-based Access Control** (RBAC) with module permissions
- ✅ **User Management** (CRUD operations for admins)
- ✅ **Device Tracking** (multi-device support)
- ✅ **Rate Limiting** (3 requests per minute for auth)
- ✅ **Failed Login Protection** (max 3 attempts per hour)
- ✅ **User Blocking System**
- ✅ **WebSocket Support** (real-time communication)
- ✅ **Swagger Documentation** (auto-generated API docs)
- ✅ **Security Headers** (Helmet.js)
- ✅ **CORS Configured**
- ✅ **Input Validation** (Joi schemas)
- ✅ **Error Logging** (file-based)
- ✅ **Database Migrations** (automatic table creation)

---

## Tech Stack

### Core
- **Runtime**: Node.js (v14+)
- **Framework**: Express.js v4.21
- **Database**: MySQL 8.0+ (via mysql2)
- **Query Builder**: Knex.js v3.1

### Authentication & Security
- **JWT**: jsonwebtoken v9.0
- **Security**: helmet v4.6
- **Rate Limiting**: express-rate-limit v5.5
- **Validation**: Joi v17.13

### Communication
- **Email**: nodemailer v6.10
- **WebSocket**: ws v8.18

### Documentation
- **API Docs**: Swagger (swagger-jsdoc, swagger-ui-express)

### Utilities
- **UUID**: uuid v8.3
- **User Agent Parser**: ua-parser-js v2.0
- **Module Aliases**: module-alias v2.2

---

## Project Structure

```
new_back/
├── config/
│   ├── index.js                    # Configuration loader
│   ├── development.env             # Development environment variables
│   ├── production.env              # Production environment variables
│   └── test.env                    # Test environment variables
│
├── src/
│   ├── api/
│   │   ├── controllers/
│   │   │   ├── authentication/
│   │   │   │   ├── code_generator.authentication.js
│   │   │   │   ├── code_verify.authentication.js
│   │   │   │   └── get_info.authentication.js
│   │   │   └── user_management/
│   │   │       ├── register_user.user_management.js
│   │   │       ├── get_user.user_management.js
│   │   │       ├── get_users.user_management.js
│   │   │       ├── update_user.user_management.js
│   │   │       ├── delete_user.user_management.js
│   │   │       └── block_user.user_management.js
│   │   │
│   │   ├── documents/              # Swagger API documentation
│   │   │   ├── authentication.documents.js
│   │   │   └── user_management.documents.js
│   │   │
│   │   ├── middlewares/
│   │   │   ├── tokenvalidator.middleware.js      # JWT validation
│   │   │   ├── isadmin.middleware.js             # Admin role check
│   │   │   ├── accessvalidator.middleware.js     # Module access check
│   │   │   ├── schemavalidator.middleware.js     # Input validation
│   │   │   └── index.js
│   │   │
│   │   ├── routes/
│   │   │   ├── authentication.routes.js
│   │   │   └── user_management.routes.js
│   │   │
│   │   └── utils/
│   │       ├── emailsender.util.js               # Email sending
│   │       ├── responsesnippet.util.js           # Response formatter
│   │       ├── catcherrorhandlersnippet.util.js  # Error handler
│   │       ├── errorlogfilegenerator.util.js     # Error logging
│   │       ├── decodedatafromtoken.util.js       # JWT decoder
│   │       ├── getclientip.util.js               # IP extractor
│   │       └── index.js
│   │
│   └── installation/
│       ├── database.installation.js              # DB connection
│       ├── initial.database.installation.js      # Table creation
│       ├── blog.database.installation.js         # Blog module tables
│       ├── swagger.installation.js               # API docs setup
│       └── websocket.installation.js             # WebSocket server
│
├── server.js                       # Main application entry point
├── package.json                    # Dependencies
└── README.md                       # This file
```

---

## Database Schema

### Tables

#### 1. `user_data`
Stores user profile information.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| username | VARCHAR (UNIQUE) | Unique username |
| name | VARCHAR | User's display name |
| ugid | VARCHAR (UNIQUE) | UUID - Universal user identifier |
| email | VARCHAR (UNIQUE) | User's email address |
| time_joined | TIMESTAMP | Registration timestamp |
| region | LONGTEXT | User's region/location |
| last_ip | VARCHAR | Last login IP address |

#### 2. `user_access`
Manages user permissions and module access.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| ugid | VARCHAR (UNIQUE) | Reference to user_data.ugid |
| authorized_modules | VARCHAR | Comma-separated module list or "all" |

**Modules**: `all`, `qr`, `analyse`, `user_management`, custom modules

#### 3. `admin_list`
List of users with admin privileges.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| ugid | VARCHAR (UNIQUE) | Reference to user_data.ugid |

#### 4. `user_blocked_list`
Blocked users (cannot login).

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| ugid | VARCHAR (UNIQUE) | Reference to user_data.ugid |

#### 5. `user_login_code`
Temporary storage for email verification codes.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| link | VARCHAR (UNIQUE) | 6-digit verification code |
| email | VARCHAR | User's email |
| agent | VARCHAR | User agent string |
| ip | VARCHAR | IP address |
| time_generated | TIMESTAMP | Code creation time |

#### 6. `user_login_failed_attempts`
Tracks failed login attempts for security.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| email | VARCHAR | User's email |
| agent | VARCHAR | User agent string |
| ip | VARCHAR | IP address |
| time_attempted | TIMESTAMP | Attempt timestamp |

#### 7. `user_devices`
Multi-device support and tracking.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| ugid | VARCHAR | Reference to user_data.ugid |
| device_id | VARCHAR | UUID for device |
| fcm_token | VARCHAR | Firebase Cloud Messaging token |
| device_name | VARCHAR | Device name (optional) |
| device_type | VARCHAR | mobile/tablet/desktop |
| last_active | TIMESTAMP | Last activity time |
| is_online | BOOLEAN | Online status |
| created_at | TIMESTAMP | Device registration time |

#### 8. `blog` (Optional)
Blog/article management.

| Column | Type | Description |
|--------|------|-------------|
| id | INT (PK) | Auto-increment primary key |
| article_id | VARCHAR (UNIQUE) | Unique article identifier |
| article_img_url | VARCHAR | Featured image URL |
| read_time | SMALLINT | Estimated read time (minutes) |
| language | VARCHAR | Article language code |
| title | VARCHAR | Article title |
| slug | VARCHAR | URL-friendly slug |
| description | VARCHAR | Short description |
| body_content | VARCHAR(5000) | Article content |
| tags | VARCHAR | Comma-separated tags |
| publish_date | VARCHAR | Publication date |
| status | VARCHAR | draft/published/archived |
| likes_count | VARCHAR | Number of likes |
| dislikes_count | VARCHAR | Number of dislikes |

---

## Authentication Flow

### 1. Code Generation (`POST /v1/authentication/codegenerator`)

```mermaid
sequenceDiagram
    User->>API: POST /codegenerator {email}
    API->>DB: Check if email exists (if admin-based)
    API->>DB: Check rate limit (3/min)
    API->>API: Generate 6-digit code
    API->>DB: Store code with IP & user-agent
    API->>Email: Send verification code
    API->>User: Success response
```

**Request Body**:
```json
{
  "email": "user@example.com"
}
```

**Rate Limiting**: 3 requests per 1 minute per IP

**Response**:
```json
{
  "success": true,
  "message": "Verification code sent successfully.",
  "data": {
    "email": "user@example.com",
    "expires_in": "15 minutes"
  }
}
```

---

### 2. Code Verification (`GET /v1/authentication/verify/:code/:email`)

```mermaid
sequenceDiagram
    User->>API: GET /verify/123456/user@example.com
    API->>DB: Validate code & email
    API->>DB: Check failed attempts (max 3/hour)
    alt User Exists
        API->>DB: Update last_ip & region
    else New User
        API->>DB: Create user with random username
        API->>DB: Create user_access record
    end
    API->>DB: Detect & save device info
    API->>API: Generate JWT token
    API->>DB: Clear verification codes
    API->>User: Return token & user data
```

**URL Parameters**:
- `code`: 6-digit verification code
- `email`: User's email address

**Query Parameters** (optional):
- `fcm_token`: Firebase Cloud Messaging token
- `device_name`: Custom device name

**Response**:
```json
{
  "success": true,
  "message": "Login successful.",
  "data": {
    "token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
    "user": {
      "ugid": "550e8400-e29b-41d4-a716-446655440000",
      "email": "user@example.com",
      "username": "user_123",
      "name": "Luke Skywalker",
      "device_id": "660e8400-e29b-41d4-a716-446655440001"
    }
  }
}
```

**JWT Token Contents**:
```json
{
  "ugid": "550e8400-e29b-41d4-a716-446655440000",
  "email": "user@example.com",
  "device_id": "660e8400-e29b-41d4-a716-446655440001",
  "iat": 1699999999,
  "exp": 1700036399
}
```

**Token Expiration**: 10 hours (configurable via `SESSION_TIME` in config)

---

### 3. Get User Info (`GET /v1/authentication/info`)

**Headers**:
```
Authorization: Bearer <jwt_token>
```

**Response**:
```json
{
  "success": true,
  "message": "User information retrieved successfully.",
  "data": {
    "ugid": "550e8400-e29b-41d4-a716-446655440000",
    "username": "user_123",
    "name": "Luke Skywalker",
    "email": "user@example.com",
    "region": "Germany",
    "time_joined": "2024-01-15T10:30:00.000Z",
    "last_ip": "192.168.1.100",
    "authorized_modules": ["qr", "analyse"],
    "is_admin": false,
    "devices": [
      {
        "device_id": "660e8400-e29b-41d4-a716-446655440001",
        "device_name": "iPhone 14 Pro",
        "device_type": "mobile",
        "is_online": true,
        "last_active": "2024-01-20T14:30:00.000Z"
      }
    ]
  }
}
```

---

## API Endpoints

### Authentication Routes (`/v1/authentication`)

| Method | Endpoint | Auth | Description |
|--------|----------|------|-------------|
| POST | `/codegenerator` | No | Generate verification code |
| GET | `/verify/:code/:email` | No | Verify code and login |
| GET | `/info` | Yes | Get authenticated user info |

---

### User Management Routes (`/v1/user_management`)

**All endpoints require authentication and admin privileges.**

| Method | Endpoint | Description | Required Fields |
|--------|----------|-------------|-----------------|
| GET | `/list` | Get all users | - |
| POST | `/register` | Register new user | email, name, authorized_modules[], is_admin |
| GET | `/:ugid` | Get user by UGID | - |
| PUT | `/:ugid` | Update user | name, authorized_modules[], is_admin |
| DELETE | `/:ugid` | Delete user | - |
| POST | `/block/:ugid` | Toggle block status | - |

#### Example: Register User

**Request**:
```bash
POST /v1/user_management/register
Authorization: Bearer <admin_token>
Content-Type: application/json

{
  "email": "newuser@example.com",
  "name": "John Doe",
  "authorized_modules": ["qr", "analyse"],
  "is_admin": false
}
```

**Response**:
```json
{
  "success": true,
  "message": "User registered successfully.",
  "data": {
    "ugid": "770e8400-e29b-41d4-a716-446655440000",
    "email": "newuser@example.com",
    "name": "John Doe",
    "username": "newuser",
    "authorized_modules": ["qr", "analyse"],
    "is_admin": false
  }
}
```

#### Example: Get All Users

**Request**:
```bash
GET /v1/user_management/list
Authorization: Bearer <admin_token>
```

**Response**:
```json
{
  "success": true,
  "message": "Users retrieved successfully.",
  "data": [
    {
      "ugid": "550e8400-e29b-41d4-a716-446655440000",
      "username": "ashkan_admin",
      "name": "Master",
      "email": "ashkan.sadri.ghamshi@gmail.com",
      "time_joined": "2024-01-01T00:00:00.000Z",
      "region": "System",
      "last_ip": "127.0.0.1",
      "authorized_modules": ["all"],
      "is_blocked": false,
      "is_admin": true
    },
    {
      "ugid": "770e8400-e29b-41d4-a716-446655440000",
      "username": "newuser",
      "name": "John Doe",
      "email": "newuser@example.com",
      "time_joined": "2024-01-20T10:00:00.000Z",
      "region": "Germany",
      "last_ip": "192.168.1.50",
      "authorized_modules": ["qr", "analyse"],
      "is_blocked": false,
      "is_admin": false
    }
  ]
}
```

---

## Installation

### Prerequisites

- Node.js v14+ and npm
- MySQL 8.0+
- SMTP server for email (or use Gmail)

### Steps

1. **Clone the repository**
```bash
git clone <repository-url>
cd new_back
```

2. **Install dependencies**
```bash
npm install
```

3. **Create MySQL database**
```sql
CREATE DATABASE immbtsgz_db_kmt CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

4. **Configure environment variables**

Edit `config/development.env`:
```env
# Project name
PROJECT_NAME="immblend"

# Server
SERVER_HOST="0.0.0.0"
SERVER_PORT=3001

# Database
DB_CLIENT="mysql2"
DB_HOST="localhost"
DB_PORT=3306
DB_USER="your_mysql_user"
DB_PASSWORD="your_mysql_password"
DB_NAME="immbtsgz_db_kmt"

# Email service
EMAIL_HOST="smtp.gmail.com"
EMAIL_PORT=465
EMAIL_SECURE=true
EMAIL_USERNAME="your_email@gmail.com"
EMAIL_PASSWORD="your_app_password"

# JWT secret (change this!)
JWT_SECRET_KEY="your_random_secret_key_here"

# Session
SESSION_SECRET="your_session_secret_here"

# Master Admin
ADMIN_MASTER_EMAIL="admin@example.com"
IS_PROJECT_ADMIN_BASED=false
```

5. **Run the server**
```bash
# Development mode (with auto-reload)
npm run dev

# Production mode
npm start
```

6. **Access the application**
- API: `http://localhost:3001`
- Swagger Docs: `http://localhost:3001/doc`
- WebSocket: `ws://localhost:3001/ws`

---

## Configuration

### Environment Variables

| Variable | Required | Default | Description |
|----------|----------|---------|-------------|
| PROJECT_NAME | No | starterkit | Project name |
| MODE | No | production | Operating mode |
| DEBUG | No | false | Enable debug logging |
| SERVER_HOST | No | 0.0.0.0 | Server host |
| SERVER_PORT | No | 3000 | Server port |
| DB_CLIENT | Yes | mysql2 | Database client |
| DB_HOST | Yes | localhost | Database host |
| DB_PORT | No | 3306 | Database port |
| DB_USER | Yes | - | Database username |
| DB_PASSWORD | Yes | - | Database password |
| DB_NAME | Yes | - | Database name |
| SESSION_TIME | No | 10 | JWT expiration (hours) |
| SESSION_SECRET | Yes | - | Session encryption key |
| JWT_SECRET_KEY | Yes | - | JWT signing key |
| EMAIL_HOST | Yes | - | SMTP server host |
| EMAIL_PORT | No | 465 | SMTP server port |
| EMAIL_SECURE | No | true | Use SSL/TLS |
| EMAIL_USERNAME | Yes | - | SMTP username |
| EMAIL_PASSWORD | Yes | - | SMTP password |
| ADMIN_MASTER_EMAIL | Yes | - | Master admin email |
| IS_PROJECT_ADMIN_BASED | No | false | Require pre-registration |

### Admin Settings

**Master Admin**:
- Email defined in `ADMIN_MASTER_EMAIL`
- Created automatically on first startup
- Has `authorized_modules = "all"`
- Cannot be deleted or blocked

**Admin-Based Mode** (`IS_PROJECT_ADMIN_BASED`):
- `false`: Anyone can register via email verification
- `true`: Only users pre-registered by admin can login

---

## Usage

### Creating a Master Admin

The master admin is created automatically on first startup. Configure the email in `config/development.env`:

```env
ADMIN_MASTER_EMAIL="admin@yourdomain.com"
```

On first run, the system will:
1. Create user with this email
2. Set `authorized_modules = "all"`
3. Add to `admin_list` table
4. Create a default device

### Adding New Users (Admin Only)

```bash
POST /v1/user_management/register
Authorization: Bearer <admin_token>

{
  "email": "user@example.com",
  "name": "User Name",
  "authorized_modules": ["qr", "analyse"],
  "is_admin": false
}
```

### User Login Flow

1. User enters email on login page
2. Frontend calls `POST /v1/authentication/codegenerator`
3. User receives 6-digit code via email
4. User enters code
5. Frontend calls `GET /v1/authentication/verify/:code/:email`
6. User receives JWT token
7. Store token in localStorage/sessionStorage
8. Include token in all subsequent requests: `Authorization: Bearer <token>`

---

## Middleware

### 1. Token Validator (`tokenvalidator.middleware.js`)

Validates JWT tokens and checks user status.

**Usage**:
```javascript
router.get('/protected', tokenvalidator, controller);
```

**Checks**:
- Valid JWT format
- Token not expired
- User exists
- User not blocked
- Updates last_ip

---

### 2. Admin Checker (`isadmin.middleware.js`)

Verifies user has admin privileges.

**Usage**:
```javascript
router.post('/admin-only', tokenvalidator, isadmin(), controller);
```

**Checks**:
- User exists in `admin_list` table

---

### 3. Access Validator (`accessvalidator.middleware.js`)

Checks module-level permissions.

**Usage**:
```javascript
router.get('/qr', tokenvalidator, accessvalidator('qr'), controller);
```

**Checks**:
- User has "all" modules, OR
- User has specific module in `authorized_modules`

---

### 4. Schema Validator (`schemavalidator.middleware.js`)

Validates request body against Joi schemas.

**Usage**:
```javascript
const schema = Joi.object({
  email: Joi.string().email().required()
});

router.post('/endpoint', schemavalidator(schema), controller);
```

---

## Security Features

### Rate Limiting

**Authentication Endpoints**:
- 3 requests per 1 minute per IP
- Prevents brute-force attacks

**General Endpoints**:
- 100 requests per 15 minutes per IP

### Failed Login Protection

- Tracks failed verification attempts
- Max 3 failures per 60 minutes
- Automatic lockout after limit
- Clears on successful login

### Security Headers (Helmet.js)

- XSS Protection
- Content Security Policy
- Frameguard (clickjacking prevention)
- HSTS (HTTP Strict Transport Security)
- No Sniff (MIME type sniffing prevention)

### CORS Configuration

- Configurable allowed origins
- Credentials support
- Preflight caching

### Input Validation

- All inputs validated with Joi schemas
- SQL injection prevention via Knex query builder
- XSS prevention via input sanitization

### Password-less Authentication

- No password storage = No password leaks
- Email-based verification codes
- Time-limited codes (15 minutes)
- One-time use codes

---

## Development

### Running in Development Mode

```bash
npm run dev
```

Features:
- Auto-reload on file changes (nodemon)
- Swagger documentation enabled
- Detailed error messages
- Debug logging

### Testing

```bash
npm test
```

Uses Jest for unit and integration testing.

### Building for Production

```bash
npm run build
```

Uses Gulp to minify and optimize code.

---

## Deployment

### Environment Setup

1. Create production environment file: `config/production.env`
2. Set `NODE_ENV=production`
3. Configure production database
4. Set strong JWT secret keys
5. Configure production email service

### Production Checklist

- [ ] Change all default passwords and secrets
- [ ] Enable HTTPS (use reverse proxy like Nginx)
- [ ] Configure firewall rules
- [ ] Set up database backups
- [ ] Configure log rotation
- [ ] Enable monitoring (e.g., PM2, New Relic)
- [ ] Set up error tracking (e.g., Sentry)
- [ ] Configure rate limiting based on traffic
- [ ] Review and update CORS origins
- [ ] Disable Swagger documentation
- [ ] Set `IS_PROJECT_ADMIN_BASED=true` if needed

### Running in Production

```bash
npm start
```

Or use a process manager like PM2:

```bash
pm2 start server.js --name "backend-api"
pm2 save
pm2 startup
```

---

## WebSocket Usage

Connect to WebSocket server:

```javascript
const ws = new WebSocket('ws://localhost:3001/ws');

ws.onopen = () => {
  console.log('Connected to WebSocket');

  // Send ping
  ws.send(JSON.stringify({ type: 'ping' }));
};

ws.onmessage = (event) => {
  const data = JSON.parse(event.data);
  console.log('Received:', data);
  // { type: 'pong', timestamp: 1699999999 }
};

ws.onerror = (error) => {
  console.error('WebSocket error:', error);
};

ws.onclose = () => {
  console.log('WebSocket connection closed');
};
```

---

## Swagger Documentation

Access at: `http://localhost:3001/doc`

Features:
- Interactive API testing
- Request/response examples
- Schema definitions
- Authentication testing (JWT)

To add new endpoints to Swagger:
1. Create documentation in `src/api/documents/<module>.documents.js`
2. Follow OpenAPI 3.0 specification
3. Restart server to reload documentation

---

## Error Handling

All errors are logged to `/logs/errors/` directory with:
- Timestamp
- Error message
- Stack trace
- Request details (if applicable)

Error response format:

```json
{
  "success": false,
  "message": "Error description",
  "data": null
}
```

---

## Module Permissions

### Available Modules

- `all` - Full access to everything
- `qr` - QR code module
- `analyse` - Analytics module
- `user_management` - User management (admin only)
- Custom modules can be added

### Assigning Modules

When creating/updating users:

```json
{
  "authorized_modules": ["qr", "analyse"]
}
```

Or grant all access:

```json
{
  "authorized_modules": ["all"]
}
```

---

## Database Migrations

Tables are created automatically on first startup. To add new columns:

1. Edit `src/installation/initial.database.installation.js`
2. Add column to table schema
3. Restart server
4. Column will be added if it doesn't exist

Example:

```javascript
{
  name: "user_data",
  columns: [
    // ... existing columns
    { new_column: "t.varchar('new_column').nullable()" }
  ]
}
```

---

## Troubleshooting

### Common Issues

**1. Database connection failed**
- Check MySQL is running
- Verify credentials in `.env` file
- Ensure database exists

**2. Email not sending**
- Verify SMTP credentials
- Check email provider settings
- Enable "Less secure apps" for Gmail
- Use App Password for Gmail with 2FA

**3. JWT token invalid**
- Check `JWT_SECRET_KEY` matches on all servers
- Verify token not expired
- Check Authorization header format: `Bearer <token>`

**4. Rate limit exceeded**
- Wait 1 minute before retrying
- Check rate limit configuration
- Verify IP address detection

**5. User not found after login**
- Check database tables created
- Verify user entry in `user_data`
- Check `user_access` table

---

## Contributing

1. Fork the repository
2. Create a feature branch
3. Commit your changes
4. Push to the branch
5. Create a Pull Request

---

## License

ISC

---

## Support

For issues and questions:
- GitHub Issues: [Repository Issues]
- Email: info@bevr.finance

---

## Changelog

### v0.0.2 (Current)
- Initial release
- Email-based authentication
- User management
- WebSocket support
- Swagger documentation

---

**Built with ❤️ by starterkit Development Team**
