// **************************
// Copyright 2023
// **************************

/**
 * Configuration Module
 *
 * This module loads environment variables and provides a structured
 * configuration object for the entire application.
 */

const path = require("path");
const dotenv = require("dotenv");

// Load environment variables based on NODE_ENV
const nodeEnv = process.env.NODE_ENV || "development";
const configPath = path.format({
  dir: path.resolve(__dirname),
  base: nodeEnv + ".env",
});
dotenv.config({ path: configPath });

// Helper functions
const toBool = (value) => value === "true" || value === true;
const getEnv = (key, defaultValue = null) =>
  process.env[key] || defaultValue;

// Get project name from environment variables
const projectName = getEnv("PROJECT_NAME", "starterkit");

//  Configuration Object
// -------------------------------

const config = {
  // Application basics
  // -----------------
  projectName,
  mode: getEnv("MODE", "production"),
  debug: toBool(getEnv("DEBUG", "false")),

  // Server configuration
  // -----------------
  server: {
    host: getEnv("SERVER_HOST", "0.0.0.0"),
    port: parseInt(process.env.PORT || getEnv("SERVER_PORT", "3000"), 10),
  },

  // Database configuration
  // -----------------
  db: {
    client: getEnv("DB_CLIENT", "mysql2"),
    host: getEnv("DB_HOST", "localhost"),
    port: parseInt(getEnv("DB_PORT", "5432"), 10),
    user: getEnv("DB_USER"),
    password: getEnv("DB_PASSWORD"),
    name: getEnv("DB_NAME"),
  },

  // Session management
  // -----------------
  session: {
    time: getEnv("SESSION_TIME", "10"),
    secret: getEnv("SESSION_SECRET", getEnv("JWT_SECRET_KEY")),
  },
  // API and security limits
  // -----------------
  limits: {
    imageUpload: {
      limit: 150, // In 30 days
    },
    link: {
      expirationTime: 0.035, // In hours
    },
    spam: {
      requestsRateLimit: 3, // Number of requests
      requetsRateLimitInterval: 1, // In minutes - It means there should be a ( requetsRateLimitInterval.value ) minute(s) gap between every ( requestsRateLimit.value ) request(s)
    },
    deviceLimit: {
      basicPlan: 3,
    },
    loginAttempts: 3,
  },
  verification: {
    codeLength: 6,
    includeLetters: true,
  },

  // Communication services
  // -----------------
  email: {
    host: getEnv("EMAIL_HOST"),
    port: parseInt(getEnv("EMAIL_PORT", "465"), 10),
    secure: toBool(getEnv("EMAIL_SECURE", "true")),
    user: getEnv("EMAIL_USERNAME"),
    pass: getEnv("EMAIL_PASSWORD"),
  },

  // Azure Blob Storage
  // -----------------
  azureStorage: {
    connectionString: getEnv("AZURE_STORAGE_CONNECTION_STRING"),
    accountName: getEnv("AZURE_STORAGE_ACCOUNT_NAME", "kmtstorage"),
  },

  // Azure Speech Service (for Live Transcription)
  // -----------------
  azureSpeech: {
    key: getEnv("AZURE_SPEECH_KEY"),
    region: getEnv("AZURE_SPEECH_REGION", "germanywestcentral"),
    endpoint: getEnv("AZURE_SPEECH_ENDPOINT"),
  },

  // Azure Language Service (for Text Analytics)
  // -----------------
  azureLanguage: {
    key: getEnv("AZURE_LANGUAGE_KEY"),
    endpoint: getEnv("AZURE_LANGUAGE_ENDPOINT"),
    region: getEnv("AZURE_LANGUAGE_REGION", "germanywestcentral"),
  },

  // Administration settings
  // -----------------
  adminSettings: {
    masterEmail: "ashkan.sadri.ghamshi@gmail.com",
    isAdminBased: false,
  },

  // External services and integrations
  // -----------------
  oauth2: {
    google: {
      clientId: getEnv("GOOGLE_CLIENT_ID"),
      clientSecret: getEnv("GOOGLE_CLIENT_SECRET"),
      redirectUri: getEnv("GOOGLE_REDIRECT_URI"),
    },
  },
};

module.exports = config;
