/**
 * ENTERPRISE-GRADE USER MIGRATION SCRIPT
 *
 * Migrates users from SQL Server (C# ASP.NET Identity) to MySQL (Node.js)
 *
 * SAFETY FEATURES:
 * - Dry-run mode (test without writing)
 * - Pre-migration verification
 * - Post-migration verification
 * - Comprehensive logging
 * - Duplicate detection
 * - Zero data loss guarantee
 *
 * USAGE:
 *   node migrations/from-csharp/migrate-users.js --dry-run  (Test mode)
 *   node migrations/from-csharp/migrate-users.js            (Execute migration)
 */

require('module-alias/register');
const sql = require('mssql');
const path = require('path');
const moduleAlias = require('module-alias');

// Setup module aliases
moduleAlias.addAliases({
  '@database': path.join(__dirname, '../../src/installation/database.installation.js'),
});

const database = require('@database');

// SQL Server Configuration
const sqlServerConfig = {
  server: 'sql-server-prezp.database.windows.net',
  port: 1433,
  database: 'prezp-prod',
  user: 'prod',
  password: 'YZbUUjP?#J@T5d',
  options: {
    encrypt: true,
    trustServerCertificate: false,
    enableArithAbort: true,
    connectionTimeout: 30000,
    requestTimeout: 30000,
  }
};

// Check if running in dry-run mode
const isDryRun = process.argv.includes('--dry-run');

// Logging utilities
const log = {
  info: (msg) => console.log(`[INFO] ${msg}`),
  success: (msg) => console.log(`[SUCCESS] ✓ ${msg}`),
  warning: (msg) => console.log(`[WARNING] ⚠ ${msg}`),
  error: (msg) => console.log(`[ERROR] ✗ ${msg}`),
  section: (msg) => console.log(`\n${'='.repeat(60)}\n${msg}\n${'='.repeat(60)}`),
};

/**
 * Verify source data from SQL Server
 */
async function verifySourceData(sqlPool) {
  log.section('STEP 1: VERIFYING SOURCE DATA (SQL Server)');

  try {
    // Get total user count
    const countResult = await sqlPool.query`
      SELECT COUNT(*) as total FROM AspNetUsers
    `;
    const totalUsers = countResult.recordset[0].total;
    log.info(`Total users in SQL Server: ${totalUsers}`);

    // Get users with sessions
    const usersWithSessions = await sqlPool.query`
      SELECT DISTINCT u.Id, u.UserName, u.Email, u.FirstName, u.LastName
      FROM AspNetUsers u
      INNER JOIN Sessions s ON s.UserId = u.Id
    `;
    log.info(`Users with sessions: ${usersWithSessions.recordset.length}`);

    // Get sample user data
    const sampleUsers = await sqlPool.query`
      SELECT TOP 5
        Id, UserName, Email, FirstName, LastName,
        EmailConfirmed, PhoneNumber, PreferredLocale
      FROM AspNetUsers
      ORDER BY Id
    `;

    log.info('\nSample user data:');
    sampleUsers.recordset.forEach(user => {
      log.info(`  - ${user.UserName} (${user.Email}) [${user.Id}]`);
    });

    // Check for null or invalid data
    const invalidData = await sqlPool.query`
      SELECT COUNT(*) as count
      FROM AspNetUsers
      WHERE Id IS NULL OR Email IS NULL OR UserName IS NULL
    `;

    if (invalidData.recordset[0].count > 0) {
      log.error(`Found ${invalidData.recordset[0].count} users with NULL required fields!`);
      throw new Error('Invalid data detected in source database');
    }

    log.success('Source data verification passed');
    return totalUsers;

  } catch (error) {
    log.error(`Source verification failed: ${error.message}`);
    throw error;
  }
}

/**
 * Verify target database (MySQL)
 */
async function verifyTargetDatabase() {
  log.section('STEP 2: VERIFYING TARGET DATABASE (MySQL)');

  try {
    // Check if tables exist
    const userDataExists = await database.schema.hasTable('user_data');
    const userAccessExists = await database.schema.hasTable('user_access');

    if (!userDataExists || !userAccessExists) {
      log.error('Required tables not found in MySQL!');
      log.error(`user_data: ${userDataExists ? 'EXISTS' : 'MISSING'}`);
      log.error(`user_access: ${userAccessExists ? 'EXISTS' : 'MISSING'}`);
      throw new Error('Target database schema incomplete');
    }

    log.success('Target tables verified: user_data, user_access');

    // Get current user count
    const currentUsers = await database('user_data').count('* as count');
    const currentCount = currentUsers[0].count;
    log.info(`Current users in MySQL: ${currentCount}`);

    // Get existing ugids
    const existingUgids = await database('user_data').select('ugid');
    log.info(`Existing UGIDs: ${existingUgids.length}`);
    if (existingUgids.length > 0) {
      log.info('Sample existing UGIDs:');
      existingUgids.slice(0, 3).forEach(u => log.info(`  - ${u.ugid}`));
    }

    log.success('Target database verification passed');
    return existingUgids.map(u => u.ugid);

  } catch (error) {
    log.error(`Target verification failed: ${error.message}`);
    throw error;
  }
}

/**
 * Fetch all users from SQL Server
 */
async function fetchSourceUsers(sqlPool) {
  log.section('STEP 3: FETCHING USERS FROM SQL SERVER');

  try {
    const result = await sqlPool.query`
      SELECT
        Id,
        UserName,
        Email,
        FirstName,
        LastName,
        EmailConfirmed,
        PhoneNumber,
        PreferredLocale,
        LockoutEnabled,
        AccessFailedCount,
        CreatedAt
      FROM AspNetUsers
      ORDER BY CreatedAt
    `;

    log.success(`Fetched ${result.recordset.length} users from SQL Server`);
    return result.recordset;

  } catch (error) {
    log.error(`Failed to fetch users: ${error.message}`);
    throw error;
  }
}

/**
 * Transform SQL Server user to MySQL format
 */
function transformUser(sqlUser) {
  const fullName = [sqlUser.FirstName, sqlUser.LastName]
    .filter(Boolean)
    .join(' ')
    .trim() || sqlUser.UserName;

  const language = sqlUser.PreferredLocale === 'de-DE' ? 'de' : 'en';

  return {
    userData: {
      ugid: sqlUser.Id,
      username: sqlUser.UserName,
      email: sqlUser.Email,
      name: fullName,
      language: language,
      time_joined: sqlUser.CreatedAt || new Date(),
    },
    userAccess: {
      ugid: sqlUser.Id,
      authorized_modules: 'all', // All migrated users get full access
    }
  };
}

/**
 * Migrate users to MySQL
 */
async function migrateUsers(users, existingUgids) {
  log.section(`STEP 4: MIGRATING USERS TO MySQL ${isDryRun ? '(DRY-RUN MODE)' : ''}`);

  const stats = {
    total: users.length,
    migrated: 0,
    skipped: 0,
    failed: 0,
    errors: []
  };

  for (const sqlUser of users) {
    try {
      const { userData, userAccess } = transformUser(sqlUser);

      // Check if user already exists
      if (existingUgids.includes(userData.ugid)) {
        log.warning(`Skipping existing user: ${userData.username} (${userData.ugid})`);
        stats.skipped++;
        continue;
      }

      if (isDryRun) {
        log.info(`[DRY-RUN] Would migrate: ${userData.username} (${userData.email})`);
        log.info(`  → ugid: ${userData.ugid}`);
        log.info(`  → name: ${userData.name}`);
        log.info(`  → language: ${userData.language}`);
        stats.migrated++;
      } else {
        // Insert into user_data
        await database('user_data').insert(userData);

        // Insert into user_access
        await database('user_access').insert(userAccess);

        log.success(`Migrated: ${userData.username} (${userData.email})`);
        stats.migrated++;
      }

    } catch (error) {
      log.error(`Failed to migrate ${sqlUser.UserName}: ${error.message}`);
      stats.failed++;
      stats.errors.push({
        user: sqlUser.UserName,
        error: error.message
      });
    }
  }

  return stats;
}

/**
 * Post-migration verification
 */
async function verifyMigration(expectedCount, existingCount) {
  log.section('STEP 5: POST-MIGRATION VERIFICATION');

  try {
    if (isDryRun) {
      log.info('[DRY-RUN] Skipping verification (no actual changes made)');
      return;
    }

    // Count users in MySQL
    const userDataCount = await database('user_data').count('* as count');
    const userAccessCount = await database('user_access').count('* as count');

    const finalUserData = userDataCount[0].count;
    const finalUserAccess = userAccessCount[0].count;

    log.info(`Users before migration: ${existingCount}`);
    log.info(`Users after migration: ${finalUserData}`);
    log.info(`Expected total: ${expectedCount + existingCount}`);
    log.info(`User access records: ${finalUserAccess}`);

    // Verify counts match
    if (finalUserData !== finalUserAccess) {
      log.error(`Mismatch! user_data (${finalUserData}) != user_access (${finalUserAccess})`);
      throw new Error('Data integrity check failed');
    }

    // Verify all ugids exist in both tables
    const userDataUgids = await database('user_data')
      .select('ugid')
      .orderBy('ugid');

    const userAccessUgids = await database('user_access')
      .select('ugid')
      .orderBy('ugid');

    const userDataSet = new Set(userDataUgids.map(u => u.ugid));
    const userAccessSet = new Set(userAccessUgids.map(u => u.ugid));

    const missingInAccess = [...userDataSet].filter(ugid => !userAccessSet.has(ugid));
    const missingInData = [...userAccessSet].filter(ugid => !userDataSet.has(ugid));

    if (missingInAccess.length > 0) {
      log.error(`UGIDs in user_data but missing in user_access: ${missingInAccess.length}`);
      missingInAccess.forEach(ugid => log.error(`  - ${ugid}`));
    }

    if (missingInData.length > 0) {
      log.error(`UGIDs in user_access but missing in user_data: ${missingInData.length}`);
      missingInData.forEach(ugid => log.error(`  - ${ugid}`));
    }

    if (missingInAccess.length === 0 && missingInData.length === 0) {
      log.success('Data integrity verified: All UGIDs match between tables');
    } else {
      throw new Error('Data integrity check failed: UGID mismatch');
    }

    // Verify sessions can now be retrieved
    const sessionsWithUsers = await database('presentation_session as ps')
      .join('user_data as ud', 'ps.ugid', 'ud.ugid')
      .count('* as count');

    const orphanedSessions = await database('presentation_session as ps')
      .leftJoin('user_data as ud', 'ps.ugid', 'ud.ugid')
      .whereNull('ud.ugid')
      .count('* as count');

    log.info(`Sessions linked to users: ${sessionsWithUsers[0].count}`);
    log.info(`Orphaned sessions (no user): ${orphanedSessions[0].count}`);

    if (orphanedSessions[0].count > 0) {
      log.warning(`Found ${orphanedSessions[0].count} orphaned sessions!`);

      // Show which UGIDs are missing
      const orphanedUgids = await database('presentation_session as ps')
        .leftJoin('user_data as ud', 'ps.ugid', 'ud.ugid')
        .whereNull('ud.ugid')
        .select('ps.ugid')
        .groupBy('ps.ugid');

      log.warning('Missing UGIDs:');
      orphanedUgids.forEach(row => log.warning(`  - ${row.ugid}`));
    } else {
      log.success('All sessions have valid user associations');
    }

    log.success('Post-migration verification passed');

  } catch (error) {
    log.error(`Verification failed: ${error.message}`);
    throw error;
  }
}

/**
 * Main migration function
 */
async function main() {
  log.section(`USER MIGRATION SCRIPT - ${isDryRun ? 'DRY-RUN MODE' : 'EXECUTION MODE'}`);

  if (isDryRun) {
    log.warning('Running in DRY-RUN mode - NO changes will be made to the database');
  } else {
    log.warning('Running in EXECUTION mode - Changes WILL be written to the database');
  }

  let sqlPool;

  try {
    // Connect to SQL Server
    log.info('Connecting to SQL Server...');
    sqlPool = await sql.connect(sqlServerConfig);
    log.success('Connected to SQL Server');

    // Step 1: Verify source data
    const totalSourceUsers = await verifySourceData(sqlPool);

    // Step 2: Verify target database
    const existingUgids = await verifyTargetDatabase();

    // Step 3: Fetch all users
    const sourceUsers = await fetchSourceUsers(sqlPool);

    // Step 4: Migrate users
    const stats = await migrateUsers(sourceUsers, existingUgids);

    // Step 5: Verify migration
    await verifyMigration(stats.migrated, existingUgids.length);

    // Final summary
    log.section('MIGRATION SUMMARY');
    log.info(`Total users in source: ${stats.total}`);
    log.success(`Successfully migrated: ${stats.migrated}`);
    log.warning(`Skipped (already exists): ${stats.skipped}`);
    if (stats.failed > 0) {
      log.error(`Failed: ${stats.failed}`);
      stats.errors.forEach(err => {
        log.error(`  - ${err.user}: ${err.error}`);
      });
    }

    if (isDryRun) {
      log.section('DRY-RUN COMPLETE');
      log.info('No changes were made to the database');
      log.info('To execute the migration, run:');
      log.info('  node migrations/from-csharp/migrate-users.js');
    } else {
      log.section('MIGRATION COMPLETE');
      log.success('All users have been migrated successfully');
    }

    process.exit(0);

  } catch (error) {
    log.section('MIGRATION FAILED');
    log.error(`Fatal error: ${error.message}`);
    log.error(error.stack);
    process.exit(1);

  } finally {
    // Cleanup
    if (sqlPool) {
      await sqlPool.close();
      log.info('SQL Server connection closed');
    }

    if (database) {
      await database.destroy();
      log.info('MySQL connection closed');
    }
  }
}

// Run migration
main();
