/**
 * Migration Script: C# SQL Server to Node.js MySQL
 *
 * This script migrates presentation session data from the C# backend (SQL Server)
 * to the Node.js backend (MySQL).
 *
 * Usage:
 *   node migrate.js                    # Run actual migration
 *   node migrate.js --dry-run          # Test without inserting data
 *   node migrate.js --batch-size=50    # Custom batch size
 *   node migrate.js --limit=100        # Limit number of records to migrate
 */

const sql = require('mssql');
const mysql = require('mysql2/promise');
const path = require('path');
const fs = require('fs');

// Parse command line arguments
const args = process.argv.slice(2);
const isDryRun = args.includes('--dry-run');
const batchSize = parseInt(args.find(arg => arg.startsWith('--batch-size='))?.split('=')[1]) || 100;
const limit = parseInt(args.find(arg => arg.startsWith('--limit='))?.split('=')[1]) || null;

// Load configuration
const configPath = path.join(__dirname, 'config.js');
if (!fs.existsSync(configPath)) {
  console.error('❌ Configuration file not found!');
  console.error('Please copy config.example.js to config.js and update with your database credentials.');
  process.exit(1);
}
const config = require('./config');

// Migration statistics
const stats = {
  totalSessions: 0,
  migratedSessions: 0,
  migratedFeedbacks: 0,
  skippedSessions: 0,
  errors: [],
  startTime: null,
  endTime: null,
};

/**
 * Connect to SQL Server (C# backend)
 */
async function connectSqlServer() {
  console.log('📡 Connecting to SQL Server...');
  try {
    await sql.connect(config.sqlServer);
    console.log('✅ Connected to SQL Server');
    return sql;
  } catch (err) {
    console.error('❌ SQL Server connection failed:', err.message);
    throw err;
  }
}

/**
 * Connect to MySQL (Node.js backend)
 */
async function connectMySQL() {
  console.log('📡 Connecting to MySQL...');
  try {
    const connection = await mysql.createConnection(config.mysql);
    console.log('✅ Connected to MySQL');
    return connection;
  } catch (err) {
    console.error('❌ MySQL connection failed:', err.message);
    throw err;
  }
}

/**
 * Validate JSON string
 */
function validateJSON(str) {
  if (!str) return null;
  try {
    JSON.parse(str);
    return str;
  } catch (e) {
    console.warn('⚠️  Invalid JSON detected, returning null');
    return null;
  }
}

/**
 * Transform TimeSpan ticks to seconds
 * SQL Server TimeSpan is stored as ticks (10,000,000 ticks = 1 second)
 */
function ticksToSeconds(ticks) {
  if (!ticks) return null;
  return ticks / 10000000;
}

/**
 * Transform C# session data to MySQL format
 */
function transformSessionData(row) {
  return {
    // Primary identifiers
    ugid: row.UserId,
    session_name: row.Name || 'Untitled Session',
    locale: row.Locale || 'en',

    // Session metadata
    presentation_id: row.PresentationId,
    recording_date: row.RecordingDate,
    session_length: row.SessionLength,
    status: 'active',
    analysis_status: 'completed',

    // Media URLs
    audio_url: row.AudioSasUri,
    video_url: row.VideoFromRecording,

    // Slide data (JSON)
    slides: validateJSON(row.Slides),
    slide_timestamps: validateJSON(row.SlideTimestamps),
    heatmap: validateJSON(row.Heatmap),
    averages: validateJSON(row.Averages),

    // Transcript data
    transcript_text: row.TranscriptText,
    transcript_words: validateJSON(row.TranscriptWords),
    transcript_sentences: validateJSON(row.TranscriptSentences),
    transcribed_segments: validateJSON(row.TranscribedSegments),
    filler_words: validateJSON(row.LookupWords),
    key_phrases: validateJSON(row.KeyPhrases),
    key_words: validateJSON(row.KeyWords),
    words_per_minute: row.WordsPerMinute,

    // Sentiment scores
    sentiment_positive_score: row.PositiveScore,
    sentiment_negative_score: row.NegativeScore,
    sentiment_neutral_score: row.NeutralScore,

    // Audio analysis data (JSON)
    audio_pitch_data: validateJSON(row.AudioPitch),
    audio_volume_data: validateJSON(row.AudioVolume),
    audio_pauses_data: validateJSON(row.AudioPauses),
    audio_timed_averages: validateJSON(row.AudioTimedAverages),

    // Audio analysis metrics
    avg_pitch: row.AveragePitch,
    avg_volume: row.AverageVolume,
    avg_pause_time: ticksToSeconds(row.AveragePauseTime),
    pauses_per_minute: row.PausesPerMinute,
    total_pause_time: ticksToSeconds(row.TotalPauseTime),

    // Session scores
    speech_speed_score: row.SpeechSpeedScore,
    pauses_score: row.PausesScore,
    repetition_score: row.RepetitionScore,
    filler_words_score: row.FillerWordsScore,
    speaking_clearly_score: row.SpeakingClearlyScore,
    overall_score: row.SessionScore,

    // Timestamps
    created_at: row.RecordingDate,
    updated_at: row.RecordingDate,
  };
}

/**
 * Validate session data before insert
 */
function validateSessionData(data) {
  const errors = [];

  if (!data.ugid) errors.push('Missing ugid');
  if (!data.session_name) errors.push('Missing session_name');
  if (!data.recording_date) errors.push('Missing recording_date');

  return {
    isValid: errors.length === 0,
    errors,
  };
}

/**
 * Fetch all sessions from SQL Server with JOIN
 */
async function fetchSessionsFromSqlServer() {
  console.log('📥 Fetching sessions from SQL Server...');

  const query = `
    SELECT TOP ${limit ? `(${limit})` : '100 PERCENT'}
      s.Id as SessionId,
      s.UserId,
      s.Name,
      s.Locale,
      s.AudioSasUri,
      s.RecordingDate,
      s.SessionLength,
      s.SessionScore,
      s.PresentationId,
      s.SlideTimestamps,
      s.Slides,
      s.TranscriptionSlides,
      s.Heatmap,
      s.Averages,
      s.VideoFromRecording,

      -- Transcription & Text Analysis
      ta.Text as TranscriptText,
      ta.Words as TranscriptWords,
      ta.Sentences as TranscriptSentences,
      ta.TranscribedSegments,
      ta.LookupWords,
      ta.KeyPhrases,
      ta.KeyWords,
      ta.WordsPerMinute,
      ta.PositiveScore,
      ta.NegativeScore,
      ta.NeutralScore,

      -- Audio Analysis
      aa.Pitch as AudioPitch,
      aa.Volume as AudioVolume,
      aa.Pauses as AudioPauses,
      aa.TimedAverages as AudioTimedAverages,
      aa.AveragePitch,
      aa.AverageVolume,
      aa.AveragePauseTime,
      aa.PausesPerMinute,
      aa.TotalPauseTime,

      -- Session Scores
      ss.SpeechSpeedScore,
      ss.PausesScore,
      ss.RepetitionScore,
      ss.FillerWordsScore,
      ss.SpeakingClearlyScore,
      ss.OverallScore

    FROM Sessions s
    LEFT JOIN SessionScores ss ON s.SessionScoresId = ss.Id
    LEFT JOIN AudioAnalysis aa ON aa.SessionId = s.Id
    LEFT JOIN Transcription t ON t.SessionId = s.Id
    LEFT JOIN TextAnalysis ta ON ta.TranscriptionId = t.Id
    ORDER BY s.RecordingDate DESC
  `;

  try {
    const result = await sql.query(query);
    stats.totalSessions = result.recordset.length;
    console.log(`✅ Fetched ${stats.totalSessions} sessions`);
    return result.recordset;
  } catch (err) {
    console.error('❌ Error fetching sessions:', err.message);
    throw err;
  }
}

/**
 * Fetch AI feedbacks for migrated sessions
 */
async function fetchAiFeedbacks() {
  console.log('📥 Fetching AI feedbacks from SQL Server...');

  const query = `
    SELECT
      af.Id,
      af.SessionId,
      af.FeedbackText,
      af.ModelVersion,
      af.CreatedAt,
      af.UpdatedAt,
      s.Locale
    FROM AiFeedbacks af
    INNER JOIN Sessions s ON af.SessionId = s.Id
    ORDER BY af.CreatedAt DESC
  `;

  try {
    const result = await sql.query(query);
    console.log(`✅ Fetched ${result.recordset.length} AI feedbacks`);
    return result.recordset;
  } catch (err) {
    console.error('❌ Error fetching AI feedbacks:', err.message);
    throw err;
  }
}

/**
 * Insert sessions into MySQL in batches
 */
async function insertSessionsToMySQL(mysqlConn, sessions) {
  console.log(`📤 ${isDryRun ? '[DRY RUN] ' : ''}Migrating ${sessions.length} sessions...`);

  const sessionIdMap = new Map(); // Map C# SessionId to MySQL id

  for (let i = 0; i < sessions.length; i += batchSize) {
    const batch = sessions.slice(i, i + batchSize);
    console.log(`Processing batch ${Math.floor(i / batchSize) + 1}/${Math.ceil(sessions.length / batchSize)}...`);

    if (isDryRun) {
      // Dry run: just validate and log
      for (const row of batch) {
        const transformed = transformSessionData(row);
        const validation = validateSessionData(transformed);

        if (!validation.isValid) {
          console.warn(`⚠️  [DRY RUN] Session ${row.SessionId} validation failed:`, validation.errors);
          stats.skippedSessions++;
        } else {
          stats.migratedSessions++;
        }
      }
    } else {
      // Actual migration with transaction
      try {
        await mysqlConn.beginTransaction();

        for (const row of batch) {
          const transformed = transformSessionData(row);
          const validation = validateSessionData(transformed);

          if (!validation.isValid) {
            console.warn(`⚠️  Skipping session ${row.SessionId}:`, validation.errors);
            stats.skippedSessions++;
            stats.errors.push({
              sessionId: row.SessionId,
              errors: validation.errors,
            });
            continue;
          }

          // Build INSERT query
          const columns = Object.keys(transformed);
          const placeholders = columns.map(() => '?').join(', ');
          const values = Object.values(transformed);

          const [result] = await mysqlConn.execute(
            `INSERT INTO presentation_session (${columns.join(', ')}) VALUES (${placeholders})`,
            values
          );

          // Store mapping for AI feedbacks migration
          sessionIdMap.set(row.SessionId, result.insertId);
          stats.migratedSessions++;
        }

        await mysqlConn.commit();
        console.log(`✅ Batch completed: ${stats.migratedSessions}/${stats.totalSessions}`);
      } catch (err) {
        await mysqlConn.rollback();
        console.error('❌ Batch failed, rolled back:', err.message);
        throw err;
      }
    }
  }

  return sessionIdMap;
}

/**
 * Insert AI feedbacks into MySQL
 */
async function insertAiFeedbacksToMySQL(mysqlConn, feedbacks, sessionIdMap) {
  if (feedbacks.length === 0) {
    console.log('ℹ️  No AI feedbacks to migrate');
    return;
  }

  console.log(`📤 ${isDryRun ? '[DRY RUN] ' : ''}Migrating ${feedbacks.length} AI feedbacks...`);

  if (isDryRun) {
    stats.migratedFeedbacks = feedbacks.length;
    return;
  }

  try {
    await mysqlConn.beginTransaction();

    for (const feedback of feedbacks) {
      const newSessionId = sessionIdMap.get(feedback.SessionId);

      if (!newSessionId) {
        console.warn(`⚠️  Skipping feedback for unknown session ${feedback.SessionId}`);
        continue;
      }

      await mysqlConn.execute(
        `INSERT INTO presentation_feedback (session_id, feedback_text, language, model_version, created_at, updated_at)
         VALUES (?, ?, ?, ?, ?, ?)`,
        [
          newSessionId,
          feedback.FeedbackText,
          feedback.Locale || 'en',
          feedback.ModelVersion,
          feedback.CreatedAt,
          feedback.UpdatedAt,
        ]
      );

      stats.migratedFeedbacks++;
    }

    await mysqlConn.commit();
    console.log(`✅ AI feedbacks migration completed: ${stats.migratedFeedbacks}`);
  } catch (err) {
    await mysqlConn.rollback();
    console.error('❌ AI feedbacks migration failed:', err.message);
    throw err;
  }
}

/**
 * Print migration summary
 */
function printSummary() {
  const duration = ((stats.endTime - stats.startTime) / 1000).toFixed(2);

  console.log('\n' + '='.repeat(60));
  console.log(`${isDryRun ? '🧪 DRY RUN ' : ''}MIGRATION SUMMARY`);
  console.log('='.repeat(60));
  console.log(`Total sessions found:      ${stats.totalSessions}`);
  console.log(`Sessions migrated:         ${stats.migratedSessions}`);
  console.log(`Sessions skipped:          ${stats.skippedSessions}`);
  console.log(`AI feedbacks migrated:     ${stats.migratedFeedbacks}`);
  console.log(`Errors:                    ${stats.errors.length}`);
  console.log(`Duration:                  ${duration}s`);
  console.log('='.repeat(60));

  if (stats.errors.length > 0) {
    console.log('\n❌ Errors:');
    stats.errors.forEach((err, index) => {
      console.log(`  ${index + 1}. Session ${err.sessionId}: ${err.errors.join(', ')}`);
    });
  }

  if (isDryRun) {
    console.log('\n✅ Dry run completed successfully! No data was inserted.');
    console.log('Run without --dry-run to perform actual migration.');
  } else {
    console.log('\n✅ Migration completed successfully!');
  }
}

/**
 * Main migration function
 */
async function runMigration() {
  stats.startTime = Date.now();

  console.log('\n' + '='.repeat(60));
  console.log(`${isDryRun ? '🧪 DRY RUN MODE - ' : ''}C# TO NODE.JS MIGRATION`);
  console.log('='.repeat(60));
  console.log(`Batch size: ${batchSize}`);
  if (limit) console.log(`Limit: ${limit} records`);
  console.log('='.repeat(60) + '\n');

  let sqlConn = null;
  let mysqlConn = null;

  try {
    // Connect to databases
    sqlConn = await connectSqlServer();
    mysqlConn = await connectMySQL();

    // Fetch data from SQL Server
    const sessions = await fetchSessionsFromSqlServer();
    const feedbacks = await fetchAiFeedbacks();

    // Migrate sessions
    const sessionIdMap = await insertSessionsToMySQL(mysqlConn, sessions);

    // Migrate AI feedbacks
    await insertAiFeedbacksToMySQL(mysqlConn, feedbacks, sessionIdMap);

    stats.endTime = Date.now();
    printSummary();
  } catch (err) {
    console.error('\n❌ Migration failed:', err);
    process.exit(1);
  } finally {
    // Close connections
    if (sqlConn) {
      await sql.close();
      console.log('\n📡 SQL Server connection closed');
    }
    if (mysqlConn) {
      await mysqlConn.end();
      console.log('📡 MySQL connection closed');
    }
  }
}

// Run migration
runMigration();
