const { jwt, uuidv4 } = require("@packages");
const database = require("@database");
const config = require("@config");
const utils = require("@utils");

const codeVerifyAuthentication = async (req, res) => {
  try {
    const { code, email } = req.params;
    const userAgent = req.get("user-agent");
    const clientIP = utils.getClientIP(req);

    if (code.length !== config.verification.codeLength) {
      await recordFailedAttempt(email, userAgent, clientIP);
      return utils.responseSnippet(
        res,
        400,
        "Invalid verification code. Please check your email or request a new code.",
        null
      );
    }

    const existingCode = await database("user_login_code")
      .where({
        link: code,
        email: email,
      })
      .first();

    if (!existingCode) {
      await recordFailedAttempt(email, userAgent, clientIP);
      return utils.responseSnippet(
        res,
        404,
        "Invalid verification code. Please check your email or request a new code.",
        null
      );
    }

    const recentFailedAttempts = await database("user_login_failed_attempts")
      .where({ email })
      .where("time_attempted", ">", database.raw("NOW() - INTERVAL ? MINUTE", [60]))
      .count("id as count")
      .first();

    const attemptCount = recentFailedAttempts ? recentFailedAttempts.count : 0;

    if (attemptCount >= config.limits.loginAttempts) {
      await database("user_login_code").where({ email }).del();
      await database("user_login_failed_attempts").where({ email }).del();

      return utils.responseSnippet(
        res,
        401,
        "Too many failed attempts. Please request a new verification code.",
        null
      );
    }

    let userUgid;
    const existingUser = await database("user_data").where({ email }).first();

    // Get language from the verification code record
    const userLanguage = existingCode.language || 'en';

    if (existingUser) {
      userUgid = existingUser.ugid;

      await database("user_data").where({ email }).update({
        last_ip: clientIP,
        language: userLanguage,
      });
    } else {
      userUgid = uuidv4();

      const starWarsNames = [
        "Luke",
        "Leia",
        "Han",
        "Chewie",
        "Vader",
        "Yoda",
        "ObiWan",
        "Anakin",
        "Padme",
        "R2D2",
        "C3PO",
        "Lando",
        "Jabba",
        "Boba",
        "Jango",
        "Mace",
        "Palpatine",
        "QuiGon",
        "Maul",
        "Rey",
        "Finn",
        "Poe",
        "Kylo",
        "BB8",
      ];

      const randomName = starWarsNames[Math.floor(Math.random() * starWarsNames.length)];
      const username = randomName + "_" + Math.floor(Math.random() * 1000);

      await database("user_data").insert({
        username,
        name: randomName,
        ugid: userUgid,
        email,
        time_joined: database.fn.now(),
        language: userLanguage,
        last_ip: clientIP,
      });

      await database("user_access").insert({
        ugid: userUgid,
        authorized_modules: null,
      });
    }

    const deviceId = uuidv4();

    const deviceInfo = getDeviceInfo(userAgent);

    const existingDevice = await database("user_devices")
      .where({
        ugid: userUgid,
        device_type: deviceInfo.deviceType,
      })
      .first();

    if (!existingDevice) {
      await database("user_devices").insert({
        ugid: userUgid,
        device_id: deviceId,
        fcm_token: req.body.fcm_token || `${deviceInfo.deviceType}_default_token`,
        device_name: deviceInfo.deviceName,
        device_type: deviceInfo.deviceType,
        is_online: true,
      });
    } else {
      await database("user_devices")
        .where({
          ugid: userUgid,
          device_type: deviceInfo.deviceType,
        })
        .update({
          is_online: true,
          last_active: database.fn.now(),
          fcm_token: req.body.fcm_token || existingDevice.fcm_token,
        });
    }

    await database("user_login_code").where({ link: code, email }).del();
    await database("user_login_failed_attempts").where({ email }).del();

    const token = jwt.sign(
      {
        ugid: userUgid,
        email,
        device_id: existingDevice ? existingDevice.device_id : deviceId,
      },
      config.session.secret,
      {
        expiresIn: config.session.time + "h",
      }
    );

    utils.responseSnippet(res, 200, "Authentication successful", {
      token,
      user: {
        email,
        ugid: userUgid,
      },
    });
  } catch (error) {
    console.error("Authentication verification error:", error);

    console.log(error);
    res.status(500).json({
      message: "A problem happened while processing your request!",
      data: null,
    });
  }
};

const getDeviceInfo = (userAgent) => {
  let deviceType = "desktop";
  let deviceName = "Unknown Desktop";

  const ua = userAgent.toLowerCase();

  if (ua.includes("mobile")) {
    deviceType = "mobile";
    if (ua.includes("iphone")) {
      deviceName = "iPhone";
    } else if (ua.includes("android")) {
      deviceName = "Android Phone";
    } else {
      deviceName = "Mobile Device";
    }
  } else if (ua.includes("tablet") || ua.includes("ipad")) {
    deviceType = "tablet";
    if (ua.includes("ipad")) {
      deviceName = "iPad";
    } else if (ua.includes("android")) {
      deviceName = "Android Tablet";
    } else {
      deviceName = "Tablet Device";
    }
  } else {
    if (ua.includes("windows")) {
      deviceName = "Windows PC";
    } else if (ua.includes("macintosh") || ua.includes("mac os")) {
      deviceName = "Mac";
    } else if (ua.includes("linux")) {
      deviceName = "Linux";
    } else {
      deviceName = "Desktop Device";
    }
  }

  return { deviceType, deviceName };
};

const recordFailedAttempt = async (email, agent, ip) => {
  const database = require("@database");

  await database("user_login_failed_attempts").insert({
    email,
    agent,
    ip,
    time_attempted: database.fn.now(),
  });
};

module.exports = codeVerifyAuthentication;
