/**
 * Delete Presentation Controller
 *
 * Handles presentation deletion from Azure Blob Storage and database
 * - Validates presentation exists
 * - Checks authorization (user can only delete their own presentations)
 * - Deletes PDF from Azure storage
 * - Removes database record
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");

const deletePresentation = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get presentation ID from params (already validated by Joi)
    const presentationId = req.params.id;

    // Fetch presentation from database
    const presentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "ugid", "file_blob_key")
      .first();

    // Check if presentation exists
    if (!presentation) {
      return utils.responseSnippet(res, 404, "Presentation not found", null);
    }

    // Authorization check: User can only delete their own presentations
    if (presentation.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to delete this presentation",
        null
      );
    }

    // Check if blob key exists
    if (!presentation.file_blob_key) {
      return utils.responseSnippet(
        res,
        500,
        "Cannot delete presentation: No file reference found",
        null
      );
    }

    const blobName = presentation.file_blob_key;
    const containerName = "presentation-materials";

    // Delete PDF from Azure Blob Storage
    const deleteResult = await azureStorage.deleteFile(containerName, blobName);

    if (!deleteResult.success) {
      return utils.responseSnippet(
        res,
        500,
        `Failed to delete presentation file: ${deleteResult.message}`,
        null
      );
    }

    // Delete presentation record from database
    await database("presentation_pdf").where({ id: presentationId }).del();

    return utils.responseSnippet(
      res,
      200,
      "Presentation deleted successfully",
      {
        deleted: true,
        id: presentationId,
      }
    );
  } catch (error) {
    console.error("Delete presentation error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = deletePresentation;
