/**
 * Delete Session Controller
 *
 * Deletes a presentation session
 * - Validates session exists
 * - Checks authorization (user can only delete their own sessions)
 * - Deletes audio/video files from Azure storage (if exist)
 * - Deletes associated AI feedback records
 * - Removes session database record
 */

const database = require("@database");
const utils = require("@utils");

const deleteSession = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get session ID from params (already validated by Joi)
    const sessionId = req.params.id;

    // Check if session exists
    const session = await database("presentation_session")
      .where({ id: sessionId })
      .first();

    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check: User can only delete their own sessions
    if (session.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to delete this session",
        null
      );
    }

    // TODO: Delete audio/video files from Azure storage if they exist
    // if (session.audio_url) {
    //   await deleteFromAzureStorage(session.audio_url);
    // }
    // if (session.video_url) {
    //   await deleteFromAzureStorage(session.video_url);
    // }

    // Delete associated AI feedback records
    await database("presentation_feedback")
      .where({ session_id: sessionId })
      .del();

    // Delete session record
    await database("presentation_session").where({ id: sessionId }).del();

    return utils.responseSnippet(
      res,
      200,
      "Session deleted successfully",
      { id: sessionId }
    );
  } catch (error) {
    console.error("Delete session error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = deleteSession;
