/**
 * Edit Presentation Controller
 *
 * Updates presentation metadata (name)
 * - Validates presentation exists
 * - Checks authorization (user can only edit their own presentations)
 * - Updates presentation name
 */

const database = require("@database");
const utils = require("@utils");

const editPresentation = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get presentation ID from params and new name from body (already validated by Joi)
    const presentationId = req.params.id;
    const { name: newName } = req.body;

    // Check if presentation exists
    const presentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "ugid")
      .first();

    if (!presentation) {
      return utils.responseSnippet(res, 404, "Presentation not found", null);
    }

    // Authorization check: User can only edit their own presentations
    if (presentation.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to edit this presentation",
        null
      );
    }

    // Update presentation name
    await database("presentation_pdf")
      .where({ id: presentationId })
      .update({
        presentation_name: newName,
      });

    // Fetch updated presentation
    const updatedPresentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "presentation_name")
      .first();

    return utils.responseSnippet(
      res,
      200,
      "Presentation updated successfully",
      updatedPresentation
    );
  } catch (error) {
    console.error("Edit presentation error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = editPresentation;
