/**
 * Request AI Feedback Controller
 *
 * Requests AI-generated feedback for a presentation session
 * - Validates session exists and belongs to user
 * - Checks if feedback already exists (cache)
 * - Generates structured AI feedback using OpenAI
 * - Stores feedback in presentation_feedback table
 * - Supports bilingual feedback (de/en)
 */

const database = require("@database");
const utils = require("@utils");
const openAIService = require("@services/openai.assistant.service");

const requestAiFeedback = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get session ID from params and regenerate flag from body (already validated by Joi)
    const session_id = req.params.id;
    const { regenerate, language: requestedLanguage } = req.body;

    console.log(`📋 [AI Feedback] Request received for session ${session_id}`);
    console.log(`📋 [AI Feedback] regenerate: ${regenerate}, language: ${requestedLanguage}`);

    // Check if session exists
    const session = await database("presentation_session")
      .where({ id: session_id })
      .first();

    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check: User can only request feedback for their own sessions
    if (session.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to request feedback for this session",
        null
      );
    }

    // Check if session analysis is complete - REMOVED (analysis_status column removed)
    // if (session.analysis_status !== "completed") {
    //   return utils.responseSnippet(
    //     res,
    //     400,
    //     "Session analysis is not yet complete. Please wait for analysis to finish.",
    //     null
    //   );
    // }

    // Detect language from request body, header, or default to German
    let language = requestedLanguage || 'de';

    // Fallback to accept-language header if not provided in body
    if (!req.body.language && req.headers['accept-language']) {
      language = req.headers['accept-language'].toLowerCase().includes('en') ? 'en' : 'de';
    }

    // Ensure valid language
    language = (language === 'en' || language === 'de') ? language : 'de';

    // Check if feedback already exists for this session
    const existingFeedback = await database("presentation_feedback")
      .where({ session_id })
      .first();

    console.log(`📋 [Cache Check] Existing feedback found:`, existingFeedback ? 'YES' : 'NO');
    if (existingFeedback) {
      console.log(`📋 [Cache Check] Feedback ID: ${existingFeedback.id}, Language in DB: ${existingFeedback.language}`);
    }

    // Determine if we should return cached feedback
    let shouldReturnCached = false;
    let cachedLanguage = null;

    if (existingFeedback && !regenerate) {
      // Parse the feedback to check its language
      try {
        const parsedFeedback = JSON.parse(existingFeedback.feedback_text);
        // Try different possible locations for language
        // First check database field, then JSON field
        cachedLanguage = existingFeedback.language || parsedFeedback.language || parsedFeedback.Language;

        // Normalize language codes (handle both uppercase and lowercase)
        if (cachedLanguage) {
          cachedLanguage = cachedLanguage.toLowerCase();
        }
        const normalizedRequestedLang = language.toLowerCase();

        console.log(`📋 [Cache Check] Parsed feedback language: ${cachedLanguage}`);
        console.log(`📋 [Cache Check] Requested language: ${normalizedRequestedLang}`);
        console.log(`📋 [Cache Check] Languages match: ${cachedLanguage === normalizedRequestedLang}`);

        // Check if the cached feedback matches the requested language
        if (cachedLanguage === normalizedRequestedLang) {
          shouldReturnCached = true;
        }
      } catch (e) {
        console.log('⚠️ Could not parse cached feedback, will generate new feedback...', e.message);
      }
    }

    // Return cached feedback if conditions are met
    if (shouldReturnCached) {
      console.log(`✅ Returning cached ${language} feedback for session ${session_id}`);
      return utils.responseSnippet(
        res,
        200,
        "AI feedback retrieved from cache",
        {
          id: existingFeedback.id,
          feedback_text: existingFeedback.feedback_text,
          language: language,
          model_version: existingFeedback.model_version,
          created_at: existingFeedback.created_at,
          cached: true
        }
      );
    }

    // Log reason for regeneration
    if (existingFeedback) {
      if (regenerate) {
        console.log(`🔄 Regenerating feedback for session ${session_id} (regenerate=true)`);
      } else if (cachedLanguage !== language) {
        console.log(`🌐 Generating new feedback in ${language} (cached was ${cachedLanguage})`);
      }
    } else {
      console.log(`🆕 Generating first feedback for session ${session_id}`);
    }

    // Extract analysis data from session
    const analysisData = openAIService.extractAnalysisData(session);

    // Validate that we have sufficient data for analysis
    if (!analysisData.transcriptionText) {
      return utils.responseSnippet(
        res,
        400,
        "Session does not have transcription data. Please ensure the session has been analyzed first.",
        null
      );
    }

    try {
      // Generate AI feedback using OpenAI
      console.log(`🤖 Generating AI feedback for session ${session_id} in language ${language}`);
      const feedbackJson = await openAIService.generateStructuredFeedbackAsync(analysisData, language);

      // Parse the feedback to ensure it's valid JSON
      let feedbackContent;
      try {
        feedbackContent = JSON.parse(feedbackJson);
      } catch (parseError) {
        console.error('Failed to parse OpenAI response as JSON:', parseError);
        // Try to extract JSON from response if it contains extra text
        const jsonMatch = feedbackJson.match(/\{[\s\S]*\}/);
        if (jsonMatch) {
          feedbackContent = JSON.parse(jsonMatch[0]);
        } else {
          throw new Error('Invalid JSON response from OpenAI');
        }
      }

      // Wrap the feedback in a response structure
      const feedbackResponse = {
        language: language,  // Use lowercase for consistency
        GeneratedAt: new Date().toISOString(),
        ModelVersion: 2, // Version 2 = structured feedback
        Content: feedbackContent
      };

      const feedbackText = JSON.stringify(feedbackResponse, null, 2);
      const modelVersion = 2;

      // Store or update feedback
      if (existingFeedback) {
        // Update existing feedback (either regenerate or different language)
        await database("presentation_feedback")
          .where({ id: existingFeedback.id })
          .update({
            feedback_text: feedbackText,
            language,
            model_version: modelVersion,
            updated_at: new Date(),
          });

        const message = regenerate
          ? "AI feedback regenerated successfully"
          : `AI feedback generated in ${language}`;

        return utils.responseSnippet(
          res,
          200,
          message,
          {
            id: existingFeedback.id,
            feedback_text: feedbackText,
            language,
            model_version: modelVersion,
            updated_at: new Date(),
          }
        );
      } else {
        // Insert new feedback (first time)
        const [feedbackId] = await database("presentation_feedback").insert({
          session_id,
          feedback_text: feedbackText,
          language,
          model_version: modelVersion,
          created_at: new Date(),
          updated_at: new Date(),
        });

        return utils.responseSnippet(
          res,
          201,
          "AI feedback generated successfully",
          {
            id: feedbackId,
            feedback_text: feedbackText,
            language,
            model_version: modelVersion,
            created_at: new Date(),
          }
        );
      }
    } catch (openAIError) {
      console.error('OpenAI Error:', openAIError.message);

      // Return a fallback response if OpenAI fails
      const fallbackText = language === 'de'
        ? 'Die KI-Feedback-Generierung ist momentan nicht verfügbar. Bitte versuchen Sie es später erneut.'
        : 'AI feedback generation is currently unavailable. Please try again later.';

      return utils.responseSnippet(
        res,
        503,
        "AI service temporarily unavailable",
        {
          feedback_text: fallbackText,
          language,
          error: true
        }
      );
    }
  } catch (error) {
    console.error("Request AI feedback error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = requestAiFeedback;
