/**
 * Get Presentation Controller
 *
 * Returns a single presentation by ID
 * - Validates presentation exists
 * - Checks authorization (user can only view their own presentations)
 * - Returns presentation metadata
 */

const database = require("@database");
const utils = require("@utils");

const getPresentation = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get presentation ID from params (already validated by Joi)
    const presentationId = req.params.id;

    // Fetch presentation from database
    const presentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "ugid", "presentation_name")
      .first();

    // Check if presentation exists
    if (!presentation) {
      return utils.responseSnippet(res, 404, "Presentation not found", null);
    }

    // Authorization check: User can only view their own presentations
    if (presentation.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to view this presentation",
        null
      );
    }

    // Remove ugid from response (no need to expose it)
    const { ugid: _, ...presentationData } = presentation;

    return utils.responseSnippet(
      res,
      200,
      "Presentation retrieved successfully",
      presentationData
    );
  } catch (error) {
    console.error("Get presentation error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = getPresentation;
