/**
 * Stream PDF File Controller (Proxy)
 *
 * Handles secure access to PDF files stored in Azure Blob Storage
 * - Validates authorization (user must own the presentation)
 * - Retrieves file_blob_key from database
 * - Streams PDF directly from Azure without buffering to memory
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");

const streamPdfFile = async (req, res) => {
  try {
    console.log('📄 === PDF FILE REQUEST ===');
    console.log('  Requested Presentation ID:', req.params.id);
    console.log('  Has req.user?', !!req.user);
    console.log('  Requesting User UGID:', req.user?.ugid || 'None');
    console.log('  Auth Header:', req.headers.authorization ? 'Present ✓' : 'Missing ✗');
    console.log('=========================');

    const { id: presentationId } = req.params;
    const requestingUserUgid = req.user ? req.user.ugid : null;

    // Validate presentation ID
    if (!presentationId) {
      return utils.responseSnippet(res, 400, "Presentation ID is required", null);
    }

    // Authorization check - user must be authenticated
    if (!requestingUserUgid) {
      return utils.responseSnippet(
        res,
        401,
        "Authentication required to view PDF files",
        null
      );
    }

    // Get presentation details from database
    const presentation = await database("presentation_pdf")
      .where({ id: presentationId })
      .select("id", "ugid", "presentation_name", "file_blob_key")
      .first();

    if (!presentation) {
      return utils.responseSnippet(res, 404, "Presentation not found", null);
    }

    // Authorization check - user must own the presentation
    if (presentation.ugid !== requestingUserUgid) {
      return utils.responseSnippet(
        res,
        403,
        "You are not authorized to view this presentation",
        null
      );
    }

    // Check if blob key exists
    if (!presentation.file_blob_key) {
      return utils.responseSnippet(
        res,
        404,
        "PDF file not found - blob key missing",
        null
      );
    }

    const containerName = "presentation-materials";
    const blobName = presentation.file_blob_key;

    // Stream file from Azure Blob Storage (proxy without buffering)
    const streamResult = await azureStorage.streamFile(
      containerName,
      blobName
    );

    if (!streamResult.success) {
      return utils.responseSnippet(
        res,
        404,
        `PDF file not found: ${streamResult.message}`,
        null
      );
    }

    // Set response headers for PDF
    res.setHeader("Content-Type", streamResult.contentType || "application/pdf");
    res.setHeader("Content-Length", streamResult.contentLength);
    res.setHeader("Content-Disposition", `inline; filename="${presentation.presentation_name}.pdf"`);
    res.setHeader("Cache-Control", "public, max-age=3600"); // Cache for 1 hour

    // Pipe Azure stream directly to client (no memory buffering!)
    streamResult.stream.pipe(res);
  } catch (error) {
    console.error("Stream PDF file error:", error);

    // Check if headers were already sent
    if (!res.headersSent) {
      return utils.catchErrorHandlerSnippet(res, error, __dirname);
    }
  }
};

module.exports = streamPdfFile;
