/**
 * Delete Profile Picture Controller
 *
 * Handles profile picture deletion from Azure Blob Storage
 * - Checks if user has a profile picture
 * - Deletes from Azure storage
 * - Updates database to remove blob key
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");

const deleteProfilePicture = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get user's profile picture key from database
    const userData = await database("user_data")
      .where({ ugid })
      .select("profile_picture_key")
      .first();

    // Check if user has a profile picture
    if (!userData || !userData.profile_picture_key) {
      return utils.responseSnippet(
        res,
        404,
        "No profile picture found to delete",
        null
      );
    }

    const containerName = "profile-images";
    const blobName = userData.profile_picture_key;

    // Delete from Azure Blob Storage
    const deleteResult = await azureStorage.deleteFile(containerName, blobName);

    if (!deleteResult.success) {
      return utils.responseSnippet(
        res,
        500,
        `Failed to delete profile picture: ${deleteResult.message}`,
        null
      );
    }

    // Update database to remove profile picture key
    await database("user_data").where({ ugid }).update({
      profile_picture_key: null,
    });

    return utils.responseSnippet(
      res,
      200,
      "Profile picture deleted successfully",
      {
        deleted: true,
      }
    );
  } catch (error) {
    console.error("Delete profile picture error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = deleteProfilePicture;
