/**
 * Get Profile Picture Controller (Proxy)
 *
 * Handles secure access to profile pictures stored in Azure Blob Storage
 * - Validates authorization (user can view their own picture)
 * - Future: Allow group members to view each other's pictures
 * - Downloads from Azure and streams to client
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");

const getProfilePicture = async (req, res) => {
  try {
    console.log('🖼️ === PROFILE PICTURE REQUEST ===');
    console.log('  Requested UGID:', req.params.ugid);
    console.log('  Has req.user?', !!req.user);
    console.log('  Requesting User UGID:', req.user?.ugid || 'None');
    console.log('  Auth Header:', req.headers.authorization ? 'Present ✓' : 'Missing ✗');
    console.log('================================');

    const { ugid } = req.params;
    const requestingUserUgid = req.user ? req.user.ugid : null;

    // Validate UGID parameter
    if (!ugid) {
      return utils.responseSnippet(res, 400, "UGID is required", null);
    }

    // Authorization check
    // Currently: User can only view their own profile picture
    // Future: Allow group members to view each other's pictures
    if (!requestingUserUgid) {
      return utils.responseSnippet(
        res,
        401,
        "Authentication required to view profile pictures",
        null
      );
    }

    // For now, users can only view their own profile picture
    // TODO: Implement group membership check for shared access
    if (requestingUserUgid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You are not authorized to view this profile picture",
        null
      );
    }

    // Get user's profile picture key from database
    const userData = await database("user_data")
      .where({ ugid })
      .select("profile_picture_key")
      .first();

    if (!userData) {
      return utils.responseSnippet(res, 404, "User not found", null);
    }

    if (!userData.profile_picture_key) {
      return utils.responseSnippet(
        res,
        404,
        "No profile picture found for this user",
        null
      );
    }

    const containerName = "profile-images";
    const blobName = userData.profile_picture_key;

    // Stream file from Azure Blob Storage (proxy without buffering)
    const streamResult = await azureStorage.streamFile(
      containerName,
      blobName
    );

    if (!streamResult.success) {
      return utils.responseSnippet(
        res,
        404,
        `Profile picture not found: ${streamResult.message}`,
        null
      );
    }

    // Set response headers
    res.setHeader("Content-Type", streamResult.contentType);
    res.setHeader("Content-Length", streamResult.contentLength);
    res.setHeader("Cache-Control", "public, max-age=3600"); // Cache for 1 hour

    // Pipe Azure stream directly to client (no memory buffering!)
    streamResult.stream.pipe(res);
  } catch (error) {
    console.error("Get profile picture error:", error);

    // Check if headers were already sent
    if (!res.headersSent) {
      return utils.catchErrorHandlerSnippet(res, error, __dirname);
    }
  }
};

module.exports = getProfilePicture;
