/**
 * Upload Profile Picture Controller
 *
 * Handles profile picture upload to Azure Blob Storage
 * - Validates file upload
 * - Deletes existing profile picture if present
 * - Uploads new picture with UGID as filename
 * - Updates database with blob key
 */

const database = require("@database");
const utils = require("@utils");
const azureStorage = require("@utils/azure.storage.util");
const path = require("path");

const uploadProfilePicture = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Validate file upload
    if (!req.file) {
      return utils.responseSnippet(res, 400, "No file uploaded", null);
    }

    const file = req.file;
    const fileExtension = path.extname(file.originalname).toLowerCase();
    const blobName = `${ugid}${fileExtension}`;
    const containerName = "profile-images";

    // Check if user already has a profile picture
    const userData = await database("user_data")
      .where({ ugid })
      .select("profile_picture_key")
      .first();

    // If user has existing profile picture, delete it from Azure
    if (userData && userData.profile_picture_key) {
      const deleteResult = await azureStorage.deleteFile(
        containerName,
        userData.profile_picture_key
      );

      if (!deleteResult.success) {
        console.warn(
          `Warning: Failed to delete old profile picture: ${deleteResult.message}`
        );
        // Continue with upload even if delete fails
      }
    }

    // Upload new profile picture to Azure
    const uploadResult = await azureStorage.uploadFile(
      containerName,
      blobName,
      file.buffer,
      {
        contentType: file.mimetype,
        metadata: {
          ugid: ugid,
          originalName: file.originalname,
          uploadedAt: new Date().toISOString(),
        },
      }
    );

    if (!uploadResult.success) {
      return utils.responseSnippet(
        res,
        500,
        `Failed to upload profile picture: ${uploadResult.message}`,
        null
      );
    }

    // Update database with new profile picture key and format
    await database("user_data").where({ ugid }).update({
      profile_picture_key: blobName,
      profile_picture_format: fileExtension.substring(1), // Remove the dot
    });

    // Return success with only key and format (frontend will construct URL)
    return utils.responseSnippet(
      res,
      200,
      "Profile picture uploaded successfully",
      {
        profile_picture_key: blobName,
        profile_picture_format: fileExtension.substring(1),
      }
    );
  } catch (error) {
    console.error("Upload profile picture error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = uploadProfilePicture;
