const userManagementDocuments = {
  list: {
    get: {
      tags: ["User Management"],
      summary: "Get all users list",
      description:
        "Retrieves list of all users with their details (requires authentication and admin access)",
      security: [
        {
          bearerAuth: [],
        },
      ],
      responses: {
        200: {
          description: "Users list retrieved successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 200,
                  },
                  message: {
                    type: "string",
                    example: "Users retrieved successfully.",
                  },
                  data: {
                    type: "array",
                    items: {
                      type: "object",
                      properties: {
                        ugid: {
                          type: "string",
                          example:
                            "3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        },
                        username: {
                          type: "string",
                          example: "johndoe",
                        },
                        name: {
                          type: "string",
                          example: "John Doe",
                        },
                        email: {
                          type: "string",
                          example: "john@example.com",
                        },
                        region: {
                          type: "string",
                          example: "System",
                        },
                        time_joined: {
                          type: "string",
                          format: "date-time",
                          example: "2025-03-27T09:15:22.123Z",
                        },
                        last_ip: {
                          type: "string",
                          example: "192.168.1.1",
                        },
                        is_admin: {
                          type: "boolean",
                          example: false,
                        },
                        is_blocked: {
                          type: "boolean",
                          example: false,
                        },
                        authorized_modules: {
                          type: "array",
                          items: {
                            type: "string",
                          },
                          example: ["qr", "analyse"],
                        },
                      },
                    },
                  },
                },
              },
            },
          },
        },
        401: {
          description: "Authentication required",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 401,
                  },
                  message: {
                    type: "string",
                    example: "Authentication required",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        403: {
          description: "Access denied",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 403,
                  },
                  message: {
                    type: "string",
                    example: "Access denied",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
      },
    },
  },

  register: {
    post: {
      tags: ["User Management"],
      summary: "Register new user",
      description:
        "Creates a new user with specified permissions (requires authentication and admin access)",
      security: [
        {
          bearerAuth: [],
        },
      ],
      requestBody: {
        required: true,
        content: {
          "application/json": {
            schema: {
              type: "object",
              required: ["email", "name", "authorized_modules"],
              properties: {
                email: {
                  type: "string",
                  format: "email",
                  example: "user@example.com",
                },
                name: {
                  type: "string",
                  example: "John Doe",
                },
                is_admin: {
                  type: "boolean",
                  example: false,
                },
                authorized_modules: {
                  type: "array",
                  items: {
                    type: "string",
                    enum: ["qr", "analyse", "user_management"],
                  },
                  example: ["qr", "analyse"],
                },
              },
            },
          },
        },
      },
      responses: {
        201: {
          description: "User created successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 201,
                  },
                  message: {
                    type: "string",
                    example: "User registered successfully.",
                  },
                  data: {
                    type: "object",
                    properties: {
                      ugid: {
                        type: "string",
                        example:
                          "3fa85f64-5717-4562-b3fc-2c963f66afa6",
                      },
                      email: {
                        type: "string",
                        example: "user@example.com",
                      },
                      name: {
                        type: "string",
                        example: "John Doe",
                      },
                      authorized_modules: {
                        type: "array",
                        items: {
                          type: "string",
                        },
                        example: ["qr", "analyse"],
                      },
                    },
                  },
                },
              },
            },
          },
        },
        400: {
          description: "Invalid input",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 400,
                  },
                  message: {
                    type: "string",
                    example: "Validation failed",
                  },
                  data: {
                    type: "object",
                    properties: {
                      errors: {
                        type: "array",
                        items: {
                          type: "object",
                          properties: {
                            field: {
                              type: "string",
                              example: "email",
                            },
                            message: {
                              type: "string",
                              example: "Invalid email format",
                            },
                          },
                        },
                      },
                    },
                  },
                },
              },
            },
          },
        },
        409: {
          description: "User already exists",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 409,
                  },
                  message: {
                    type: "string",
                    example: "User already exists.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
      },
    },
  },

  "{ugid}": {
    put: {
      tags: ["User Management"],
      summary: "Update user",
      description:
        "Updates user information including name and permissions",
      parameters: [
        {
          in: "path",
          name: "ugid",
          schema: {
            type: "string",
          },
          required: true,
          description: "User unique identifier",
        },
      ],
      security: [
        {
          bearerAuth: [],
        },
      ],
      requestBody: {
        required: true,
        content: {
          "application/json": {
            schema: {
              type: "object",
              required: ["name", "is_admin"],
              properties: {
                name: {
                  type: "string",
                  example: "John Doe",
                },
                is_admin: {
                  type: "boolean",
                  example: false,
                },
                authorized_modules: {
                  type: "array",
                  items: {
                    type: "string",
                    enum: ["qr", "analyse", "user_management"],
                  },
                  example: ["qr", "analyse"],
                },
              },
            },
          },
        },
      },
      responses: {
        200: {
          description: "User updated successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 200,
                  },
                  message: {
                    type: "string",
                    example: "User updated successfully.",
                  },
                  data: {
                    type: "object",
                    properties: {
                      ugid: {
                        type: "string",
                        example:
                          "3fa85f64-5717-4562-b3fc-2c963f66afa6",
                      },
                      name: {
                        type: "string",
                        example: "John Doe",
                      },
                      is_admin: {
                        type: "boolean",
                        example: false,
                      },
                      authorized_modules: {
                        type: "array",
                        items: {
                          type: "string",
                        },
                        example: ["qr", "analyse"],
                      },
                    },
                  },
                },
              },
            },
          },
        },
        404: {
          description: "User not found",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 404,
                  },
                  message: {
                    type: "string",
                    example: "User not found.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
      },
    },
    delete: {
      tags: ["User Management"],
      summary: "Delete user",
      description: "Deletes a user from the system",
      parameters: [
        {
          in: "path",
          name: "ugid",
          schema: {
            type: "string",
          },
          required: true,
          description: "User unique identifier",
        },
      ],
      security: [
        {
          bearerAuth: [],
        },
      ],
      responses: {
        200: {
          description: "User deleted successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 200,
                  },
                  message: {
                    type: "string",
                    example: "User deleted successfully.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        403: {
          description: "Cannot delete admin user",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 403,
                  },
                  message: {
                    type: "string",
                    example: "Cannot delete admin user.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        404: {
          description: "User not found",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 404,
                  },
                  message: {
                    type: "string",
                    example: "User not found.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
      },
    },
  },

  "block/{ugid}": {
    post: {
      tags: ["User Management"],
      summary: "Block/Unblock user",
      description: "Toggles user block status",
      parameters: [
        {
          in: "path",
          name: "ugid",
          schema: {
            type: "string",
          },
          required: true,
          description: "User unique identifier",
        },
      ],
      security: [
        {
          bearerAuth: [],
        },
      ],
      responses: {
        200: {
          description: "User block status updated successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 200,
                  },
                  message: {
                    type: "string",
                    example:
                      "User block status updated successfully.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        403: {
          description: "Cannot block admin user",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 403,
                  },
                  message: {
                    type: "string",
                    example: "Cannot block admin user.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        404: {
          description: "User not found",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  status: {
                    type: "integer",
                    example: 404,
                  },
                  message: {
                    type: "string",
                    example: "User not found.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
      },
    },
  },
};

module.exports = userManagementDocuments;
