/**
 * Flexible Token Validator Middleware
 *
 * This middleware checks for JWT tokens in both:
 * 1. Authorization header (Bearer token) - standard method
 * 2. Query parameter (?token=...) - for cases where headers can't be set (like video tags)
 *
 * Used specifically for endpoints that need to support both authentication methods
 */

const jwt = require("jsonwebtoken");
const config = require("@config");
const utils = require("@utils");
const database = require("@database");

const flexibleTokenValidator = async (req, res, next) => {
  try {
    let token = null;

    // First, check Authorization header
    const authHeader = req.headers.authorization;
    if (authHeader && authHeader.startsWith("Bearer ")) {
      token = authHeader.split(" ")[1];
      console.log('🔑 Token found in Authorization header');
    }

    // If no header token, check query parameter
    if (!token && req.query.token) {
      token = req.query.token;
      console.log('🔑 Token found in query parameter');
    }

    // If still no token, return 401
    if (!token) {
      return utils.responseSnippet(
        res,
        401,
        "Authentication required. Please provide a valid token.",
        null
      );
    }

    // Verify the token
    const decodedToken = jwt.verify(token, config.session.secret);

    // Check if user is blocked
    const blockedUser = await database("user_blocked_list")
      .where({ ugid: decodedToken.ugid })
      .first();

    if (blockedUser) {
      return utils.responseSnippet(
        res,
        403,
        "Your account has been blocked. Please contact support.",
        null
      );
    }

    // Get user data
    const user = await database("user_data")
      .where({ ugid: decodedToken.ugid })
      .first();

    if (!user) {
      return utils.responseSnippet(
        res,
        401,
        "User not found. Please login again.",
        null
      );
    }

    // Check if device is still online (not logged out)
    if (decodedToken.device_id) {
      const device = await database("user_devices")
        .where({
          ugid: decodedToken.ugid,
          device_id: decodedToken.device_id,
        })
        .first();

      if (device && !device.is_online) {
        return utils.responseSnippet(
          res,
          401,
          "Session has been terminated. Please login again.",
          null
        );
      }

      // Update device last_active timestamp
      if (device) {
        await database("user_devices")
          .where({
            ugid: decodedToken.ugid,
            device_id: decodedToken.device_id,
          })
          .update({
            last_active: database.fn.now(),
          });
      }
    }

    // Set user in request
    req.user = {
      ...decodedToken,
      name: user.name,
      email: user.email,
    };

    // Update last IP
    await database("user_data")
      .where({ ugid: decodedToken.ugid })
      .update({ last_ip: utils.getClientIP(req) });

    next();
  } catch (error) {
    if (error.name === "JsonWebTokenError") {
      return utils.responseSnippet(
        res,
        401,
        "Invalid token. Please login again.",
        null
      );
    }

    if (error.name === "TokenExpiredError") {
      return utils.responseSnippet(
        res,
        401,
        "Token expired. Please login again.",
        null
      );
    }

    return utils.catchErrorHandlerSnippet(res, error, __filename);
  }
};

module.exports = flexibleTokenValidator;