const jwt = require("jsonwebtoken");
const config = require("@config");
const utils = require("@utils");
const database = require("@database");

const tokenvalidator = async (req, res, next) => {
  try {
    const authHeader = req.headers.authorization;

    if (!authHeader || !authHeader.startsWith("Bearer ")) {
      return utils.responseSnippet(
        res,
        401,
        "Authentication required. Please provide a valid token.",
        null
      );
    }

    const token = authHeader.split(" ")[1];
    const decodedToken = jwt.verify(token, config.session.secret);

    const blockedUser = await database("user_blocked_list")
      .where({ ugid: decodedToken.ugid })
      .first();

    if (blockedUser) {
      return utils.responseSnippet(
        res,
        403,
        "Your account has been blocked. Please contact support.",
        null
      );
    }

    const user = await database("user_data")
      .where({ ugid: decodedToken.ugid })
      .first();

    if (!user) {
      return utils.responseSnippet(
        res,
        401,
        "User not found. Please login again.",
        null
      );
    }

    // Check if device is still online (not logged out)
    if (decodedToken.device_id) {
      const device = await database("user_devices")
        .where({
          ugid: decodedToken.ugid,
          device_id: decodedToken.device_id,
        })
        .first();

      if (device && !device.is_online) {
        return utils.responseSnippet(
          res,
          401,
          "Session has been terminated. Please login again.",
          null
        );
      }

      // Update device last_active timestamp
      if (device) {
        await database("user_devices")
          .where({
            ugid: decodedToken.ugid,
            device_id: decodedToken.device_id,
          })
          .update({
            last_active: database.fn.now(),
          });
      }
    }

    req.user = {
      ...decodedToken,
      name: user.name,
      email: user.email,
    };

    await database("user_data")
      .where({ ugid: decodedToken.ugid })
      .update({ last_ip: utils.getClientIP(req) });

    next();
  } catch (error) {
    if (error.name === "JsonWebTokenError") {
      return utils.responseSnippet(
        res,
        401,
        "Invalid token. Please login again.",
        null
      );
    }

    if (error.name === "TokenExpiredError") {
      return utils.responseSnippet(
        res,
        401,
        "Token expired. Please login again.",
        null
      );
    }

    return utils.catchErrorHandlerSnippet(res, error, __filename);
  }
};

module.exports = tokenvalidator;
