/**
 * Presentation Routes - Unified API
 *
 * Handles both PDF management and session management routes
 * Mounted at /api/presentation
 */

const { express, Joi } = require("@packages");
const { schemavalidator, tokenvalidator, flexibleTokenValidator } = require("@middlewares");
const { pdfUpload, videoUpload, handleMulterError } = require("@middlewares/multer.middleware");
const presentationRouter = express.Router();

// ============================================================================
// PDF MANAGEMENT ENDPOINTS
// ============================================================================

/**
 * POST /api/presentation/pdf
 * Upload a new presentation PDF
 * - Requires authentication
 * - Accepts multipart/form-data with 'File' field
 * - Max file size: 20MB
 * - Allowed format: PDF only
 */
presentationRouter.post(
  "/pdf",
  tokenvalidator,
  pdfUpload.single("File"),
  handleMulterError,
  (req, res) => {
    require("../controllers/presentation/add.pdf.js")(req, res);
  }
);

/**
 * GET /api/presentation/pdf
 * Get all presentations for the authenticated user
 * - Requires authentication
 * - Returns list of user's presentations
 */
presentationRouter.get(
  "/pdf",
  schemavalidator.SchemaValidator("body", Joi.object({}).unknown(false)),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/list.pdf.js")(req, res);
  }
);

/**
 * GET /api/presentation/pdf/:id
 * Get a single presentation by ID
 * - Requires authentication
 * - Authorization: User can only view their own presentations
 * - Returns presentation details
 */
presentationRouter.get(
  "/pdf/:id",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Presentation ID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/get.pdf.js")(req, res);
  }
);

/**
 * GET /api/presentation/pdf/:id/file
 * Stream PDF file content (Proxy)
 * - Requires authentication
 * - Authorization: User can only view their own presentations
 * - Streams PDF directly from Azure Blob Storage
 * - No buffering to memory (efficient for large files)
 * - Sets appropriate headers for PDF viewing
 */
presentationRouter.get(
  "/pdf/:id/file",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Presentation ID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/stream.pdf.file.js")(req, res);
  }
);

/**
 * PUT /api/presentation/pdf/:id
 * Edit presentation metadata (name)
 * - Requires authentication
 * - Authorization: User can only edit their own presentations
 * - Updates presentation name
 */
presentationRouter.put(
  "/pdf/:id",
  schemavalidator.SchemaValidator(
    "body",
    Joi.object({
      name: Joi.string().min(1).max(255).label("Presentation Name").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/edit.pdf.js")(req, res);
  }
);

/**
 * DELETE /api/presentation/pdf/:id
 * Delete a presentation
 * - Requires authentication
 * - Authorization: User can only delete their own presentations
 * - Deletes PDF from Azure storage
 * - Removes database record
 */
presentationRouter.delete(
  "/pdf/:id",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Presentation ID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/delete.pdf.js")(req, res);
  }
);

// ============================================================================
// SESSION MANAGEMENT ENDPOINTS
// ============================================================================

/**
 * GET /api/presentation/sessions
 * Get all sessions for the authenticated user
 * - Requires authentication
 * - Returns list of user's sessions (empty array if none)
 * - Sorted by recording_date DESC
 */
presentationRouter.get(
  "/sessions",
  schemavalidator.SchemaValidator("body", Joi.object({}).unknown(false)),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/list.session.js")(req, res);
  }
);

/**
 * POST /api/presentation/sessions
 * Create a new presentation session with video upload and analysis data
 * - Requires authentication
 * - Accepts multipart/form-data with 'video' field
 * - Max file size: 100MB
 * - Allowed formats: WebM, MP4
 * - Stores video in Azure Blob Storage
 * - Saves all real-time analysis data
 * - Returns session ID and details
 */
presentationRouter.post(
  "/sessions",
  (req, res, next) => {
    console.log('🎯 === POST /api/presentation/sessions REQUEST RECEIVED ===');
    console.log('📋 Headers:', {
      'content-type': req.headers['content-type'],
      'authorization': req.headers.authorization ? 'Present ✓' : 'Missing ✗',
    });
    console.log('📦 Body keys:', Object.keys(req.body || {}).length);
    console.log('📁 Files:', req.files || req.file ? 'Present ✓' : 'Not yet (before multer)');
    console.log('================================================================');
    next();
  },
  (req, res, next) => {
    console.log('🔐 [Step 1] Before token validation...');
    next();
  },
  tokenvalidator,
  (req, res, next) => {
    console.log('✅ [Step 2] Token validated, user:', req.user?.ugid || 'UNKNOWN');
    console.log('📤 [Step 3] Starting multer video upload...');
    next();
  },
  videoUpload.single("video"),
  (req, res, next) => {
    console.log('✅ [Step 4] Multer completed!');
    console.log('📁 Video file:', req.file ? `${req.file.originalname} (${req.file.size} bytes)` : '❌ NO FILE UPLOADED!');
    console.log('📋 Body fields:', Object.keys(req.body || {}).length, 'fields');
    next();
  },
  handleMulterError,
  (req, res) => {
    console.log('🚀 === Passing to controller ===');
    console.log('📁 Final video check:', req.file ? 'Present ✓' : '❌ MISSING');
    console.log('================================');
    require("../controllers/presentation/create.session.js")(req, res);
  }
);

/**
 * GET /api/presentation/sessions/:id
 * Get a single session by ID
 * - Requires authentication
 * - Authorization: User can only view their own sessions
 * - Returns session details with all analysis data
 * - Includes AI feedback if available
 */
presentationRouter.get(
  "/sessions/:id",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Session ID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/get.session.js")(req, res);
  }
);

/**
 * GET /api/presentation/sessions/:id/video
 * Stream video file for a session (Proxy)
 * - Requires authentication (supports both header and query parameter token)
 * - Authorization: User can only view their own session videos
 * - Streams video directly from Azure Blob Storage
 * - Supports range requests for seeking
 * - No buffering to memory (efficient for large files)
 */
presentationRouter.get(
  "/sessions/:id/video",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Session ID").required(),
    }).unknown(false)
  ),
  flexibleTokenValidator, // Use flexible validator to support query parameter token
  (req, res) => {
    require("../controllers/presentation/stream.session.video.js")(req, res);
  }
);

/**
 * PUT /api/presentation/sessions/:id/rename
 * Rename a session
 * - Requires authentication
 * - Authorization: User can only rename their own sessions
 * - Updates session_name field
 */
presentationRouter.put(
  "/sessions/:id/rename",
  schemavalidator.SchemaValidator(
    "body",
    Joi.object({
      session_name: Joi.string().min(1).max(255).label("Session Name").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/rename.session.js")(req, res);
  }
);

/**
 * DELETE /api/presentation/sessions/:id
 * Delete a session
 * - Requires authentication
 * - Authorization: User can only delete their own sessions
 * - Deletes audio/video from Azure storage (if exists)
 * - Removes database record and associated AI feedback
 */
presentationRouter.delete(
  "/sessions/:id",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      id: Joi.number().integer().positive().label("Session ID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/delete.session.js")(req, res);
  }
);

/**
 * POST /api/presentation/sessions/:id/feedback
 * Request AI-generated feedback for a session
 * - Requires authentication
 * - Authorization: User can only request feedback for their own sessions
 * - Checks user quota before generating
 * - Uses language from locale setting (en/de)
 * - Stores feedback in presentation_feedback table
 */
presentationRouter.post(
  "/sessions/:id/feedback",
  schemavalidator.SchemaValidator(
    "body",
    Joi.object({
      regenerate: Joi.boolean().label("Regenerate").default(false),
      language: Joi.string().valid('de', 'en').label("Language").default('de')
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/feedback.session.js")(req, res);
  }
);

/**
 * PUT /api/presentation/sessions/:id/data
 * Update session analysis data
 * - Requires authentication
 * - Authorization: User can only update their own sessions
 * - Updates transcript, audio analysis, scores, etc.
 * - Used by analysis service after processing
 */
presentationRouter.put(
  "/sessions/:id/data",
  schemavalidator.SchemaValidator(
    "body",
    Joi.object({
      // Transcript data (optional)
      transcript_text: Joi.string().allow(null).optional(),
      transcript_words: Joi.string().allow(null).optional(), // JSON string
      transcript_sentences: Joi.string().allow(null).optional(),
      filler_words: Joi.string().allow(null).optional(),
      key_phrases: Joi.string().allow(null).optional(),
      words_per_minute: Joi.number().allow(null).optional(),

      // Sentiment scores (optional)
      sentiment_positive_score: Joi.number().allow(null).optional(),
      sentiment_negative_score: Joi.number().allow(null).optional(),
      sentiment_neutral_score: Joi.number().allow(null).optional(),

      // Audio analysis (optional)
      audio_pitch_data: Joi.string().allow(null).optional(),
      audio_volume_data: Joi.string().allow(null).optional(),
      audio_pauses_data: Joi.string().allow(null).optional(),
      avg_pitch: Joi.number().allow(null).optional(),
      avg_volume: Joi.number().allow(null).optional(),
      avg_pause_time: Joi.number().allow(null).optional(),
      pauses_per_minute: Joi.number().allow(null).optional(),

      // Session scores (optional)
      speech_speed_score: Joi.number().allow(null).optional(),
      pauses_score: Joi.number().allow(null).optional(),
      repetition_score: Joi.number().allow(null).optional(),
      filler_words_score: Joi.number().allow(null).optional(),
      speaking_clearly_score: Joi.number().allow(null).optional(),
      overall_score: Joi.number().allow(null).optional(),

      // Session status - REMOVED: analysis_status column removed
      // analysis_status: Joi.string()
      //   .valid("pending", "processing", "completed", "failed")
      //   .label("Analysis Status")
      //   .optional(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/presentation/update.session.js")(req, res);
  }
);

module.exports = presentationRouter;
