/**
 * Users Routes
 *
 * Handles user profile-related routes including profile picture management
 * These routes are mounted at /api/users and /v1/users
 */

const { express, Joi } = require("@packages");
const { schemavalidator, tokenvalidator } = require("@middlewares");
const { profilePictureUpload, handleMulterError } = require("@middlewares/multer.middleware");
const usersRouter = express.Router();

/**
 * POST /api/users/UploadProfilePicture
 * Upload user profile picture
 * - Requires authentication
 * - Accepts multipart/form-data with 'image' field
 * - Max file size: 5MB
 * - Allowed formats: PNG, JPG/JPEG only
 */
usersRouter.post(
  "/UploadProfilePicture",
  tokenvalidator,
  profilePictureUpload.single("image"), // Frontend sends 'image' field
  handleMulterError,
  (req, res) => {
    require("../controllers/user_profile/upload_profile_picture.user_profile.js")(req, res);
  }
);

/**
 * DELETE /api/users/RemoveProfilePicture
 * Remove user profile picture
 * - Requires authentication
 * - Deletes picture from Azure storage
 * - Updates database to remove picture reference
 */
usersRouter.delete(
  "/RemoveProfilePicture",
  schemavalidator.SchemaValidator("body", Joi.object({}).unknown(false)),
  tokenvalidator,
  (req, res) => {
    require("../controllers/user_profile/delete_profile_picture.user_profile.js")(req, res);
  }
);

/**
 * GET /api/users/ProfilePicture/:ugid
 * Get user profile picture (proxy endpoint)
 * - Requires authentication
 * - Authorization: User can view their own picture (future: group members)
 * - Streams image from Azure storage
 */
usersRouter.get(
  "/ProfilePicture/:ugid",
  schemavalidator.SchemaValidator(
    "params",
    Joi.object({
      ugid: Joi.string().label("UGID").required(),
    }).unknown(false)
  ),
  tokenvalidator,
  (req, res) => {
    require("../controllers/user_profile/get_profile_picture.user_profile.js")(req, res);
  }
);

module.exports = usersRouter;
