// Imports
const utils = require("@utils");
const config = require("@config");
const database = require("./database.installation");

// Packages
const { v4: uuidv4 } = require("uuid");

// Create DB tables and add admin user
const createBackendDatabaseTablesAndCols = async () => {
  try {
    await database.raw("SELECT 1");
    await createTablesAndCols();
  } catch (error) {
    console.error("Database initialization error:", error);
    throw error;
  }
};

// Create tables and columns
const createTablesAndCols = async () => {
  for (const table of tablesWithColumns) {
    try {
      const exists = await database.schema.hasTable(table.name);

      if (!exists) {
        await database.schema.createTable(table.name, (t) => {
          console.log(`* TABLE CREATION : ${table.name} was created`);

          table.columns.forEach((column) => {
            const columnName = Object.keys(column)[0];
            const columnDef = column[columnName];

            const parts = columnDef
              .replace("t.", "")
              .match(/(\w+)\('([^']+)'(?:,\s*(\d+))?\)(.+)?/);

            if (parts) {
              let [, type, name, length, modifiers = ""] = parts;

              let col;
              switch (type) {
                case "increments":
                  col = t.increments(name);
                  break;
                case "varchar":
                  col = t.string(name, length || 255);
                  break;
                case "longtext":
                  col = t.text(name, "longtext");
                  break;
                case "timestamp":
                  col = t.timestamp(name);
                  break;
                default:
                  col = t[type](name);
              }

              if (modifiers.includes(".unique()")) {
                col.unique();
              }
              if (modifiers.includes(".nullable()")) {
                col.nullable();
              }
              if (modifiers.includes("defaultTo(database.fn.now())")) {
                col.defaultTo(database.fn.now());
              }
            }

            console.log(`- COLUMN CREATION : ${columnName} added to ${table.name} table`);
          });
        });
      } else {
        for (const column of table.columns) {
          const columnName = Object.keys(column)[0];
          await addColumnIfNotExists(table.name, columnName, column[columnName]);
        }
      }
    } catch (error) {
      console.error(`Error processing table ${table.name}:`, error);
      throw error;
    }
  }

  await createMasterAdminUser();
  await createMasterAdminDevice();
};

// Add column if not exists
const addColumnIfNotExists = async (tableName, columnName, columnDef) => {
  try {
    const hasColumn = await database.schema.hasColumn(tableName, columnName);
    if (!hasColumn) {
      await database.schema.alterTable(tableName, (t) => {
        const parts = columnDef
          .replace("t.", "")
          .match(/(\w+)\('([^']+)'(?:,\s*(\d+))?\)(.+)?/);

        if (parts) {
          let [, type, , length, modifiers = ""] = parts;

          let col;
          switch (type) {
            case "increments":
              col = t.increments(columnName);
              break;
            case "varchar":
              col = t.string(columnName, length || 255);
              break;
            case "longtext":
              col = t.text(columnName, "longtext");
              break;
            case "timestamp":
              col = t.timestamp(columnName);
              break;
            default:
              col = t[type](columnName);
          }

          if (modifiers.includes(".unique()")) {
            col.unique();
          }
          if (modifiers.includes(".nullable()")) {
            col.nullable();
          }
          if (modifiers.includes("defaultTo(database.fn.now())")) {
            col.defaultTo(database.fn.now());
          }
        }
        console.log(`- COLUMN CREATION : ${columnName} added to ${tableName}`);
      });
    }
  } catch (error) {
    console.error(`Error adding column ${columnName} to ${tableName}:`, error);
    throw error;
  }
};

// Create master admin user
const createMasterAdminUser = async () => {
  try {
    const masterEmail = config.adminSettings.masterEmail;

    if (!masterEmail) {
      console.warn("- ADMIN CREATION : No master email provided in config");
      return;
    }

    const existingUser = await database("user_data")
      .where({ email: masterEmail })
      .first();

    if (existingUser) {
      const existingAdmin = await database("user_access")
        .where({ ugid: existingUser.ugid })
        .first();

      if (!existingAdmin) {
        await database("user_access").insert({
          ugid: existingUser.ugid,
          authorized_modules: "all",
        });
        console.log(
          `- ADMIN CREATION : Added existing user ${masterEmail} to admin list`
        );
      }
      
      // Check if user is already in admin_list
      const existingInAdminList = await database("admin_list")
        .where({ ugid: existingUser.ugid })
        .first();
        
      if (!existingInAdminList) {
        // Add to admin_list table
        await database("admin_list").insert({
          ugid: existingUser.ugid,
        });
        console.log(`- ADMIN CREATION : Added existing user ${masterEmail} to admin_list`);
      }
    } else {
      const ugid = uuidv4();
      const username = masterEmail.split("@")[0] + "_admin";

      await database("user_data").insert({
        username: username,
        name: "Master",
        ugid: ugid,
        email: masterEmail,
        time_joined: database.fn.now(),
        language: "en",
      });

      console.log(`- ADMIN CREATION : Created master admin user ${masterEmail}`);

      await database("user_access").insert({
        ugid: ugid,
        authorized_modules: "all",
      });

      console.log(`- ADMIN CREATION : Added ${masterEmail} to admin list`);
      
      // Add to admin_list table
      await database("admin_list").insert({
        ugid: ugid,
      });
      
      console.log(`- ADMIN CREATION : Added ${masterEmail} to admin_list`);
    }
  } catch (error) {
    console.error("Error creating master admin user:", error);
  }
};

// Create admin device
const createMasterAdminDevice = async () => {
  try {
    const masterEmail = config.adminSettings.masterEmail;

    if (!masterEmail) {
      console.log("- DEVICE CREATION : No master email provided in config");
      return;
    }

    const masterUser = await database("user_data").where({ email: masterEmail }).first();

    if (!masterUser) {
      console.log("- DEVICE CREATION : Master admin user not found");
      return;
    }

    const existingDevice = await database("user_devices")
      .where({ ugid: masterUser.ugid })
      .first();

    if (!existingDevice) {
      const deviceId = uuidv4();

      await database("user_devices").insert({
        ugid: masterUser.ugid,
        device_id: deviceId,
        fcm_token: "admin_default_token",
        device_name: "Admin Default Device",
        device_type: "web",
        is_online: true,
      });

      console.log(
        `- DEVICE CREATION : Created default device for admin user ${masterEmail}`
      );
    }
  } catch (error) {
    console.error("Error creating master admin device:", error);
  }
};

// Tables schema
const tablesWithColumns = [
  {
    name: "user_data",
    columns: [
      { id: "t.increments('id')" },
      { username: "t.varchar('username').unique()" },
      { name: "t.varchar('name')" },
      { ugid: "t.varchar('ugid').unique()" },
      { email: "t.varchar('email').unique()" },
      { time_joined: "t.timestamp('time_joined')" },
      { language: "t.varchar('language', 5)" },
      { profile_picture_key: "t.varchar('profile_picture_key', 255).nullable()" },
      { profile_picture_format: "t.varchar('profile_picture_format', 10).nullable()" },
      { last_ip: "t.varchar('last_ip').nullable()" },
    ],
  },
  {
    name: "user_access",
    columns: [
      { id: "t.increments('id')" },
      { ugid: "t.varchar('ugid').unique()" },
      { authorized_modules: "t.varchar('authorized_modules')" },
    ],
  },
  {
    name: "user_blocked_list",
    columns: [{ id: "t.increments('id')" }, { ugid: "t.varchar('ugid').unique()" }],
  },
  {
    name: "admin_list",
    columns: [{ id: "t.increments('id')" }, { ugid: "t.varchar('ugid').unique()" }],
  },
  {
    name: "user_login_code",
    columns: [
      { id: "t.increments('id')" },
      { link: "t.varchar('link').unique()" },
      { email: "t.varchar('email')" },
      { agent: "t.varchar('agent')" },
      { ip: "t.varchar('ip')" },
      { language: "t.varchar('language', 5)" },
      { time_generated: "t.timestamp('time_generated')" },
    ],
  },
  {
    name: "user_login_failed_attempts",
    columns: [
      { id: "t.increments('id')" },
      { email: "t.varchar('email')" },
      { agent: "t.varchar('agent')" },
      { ip: "t.varchar('ip')" },
      { time_attempted: "t.timestamp('time_attempted')" },
    ],
  },
  {
    name: "user_devices",
    columns: [
      { id: "t.increments('id')" },
      { ugid: "t.varchar('ugid')" },
      { device_id: "t.varchar('device_id')" },
      { fcm_token: "t.varchar('fcm_token')" },
      { device_name: "t.varchar('device_name').nullable()" },
      { device_type: "t.varchar('device_type')" },
      { last_active: "t.timestamp('last_active').defaultTo(database.fn.now())" },
      { is_online: "t.boolean('is_online').defaultTo(false)" },
      { created_at: "t.timestamp('created_at').defaultTo(database.fn.now())" },
    ],
  },
  // Note: presentation tables (presentation_pdf, presentation_session, presentation_feedback)
  // are defined in presentation.installation.js
];

module.exports = createBackendDatabaseTablesAndCols;
