/**
 * Fix Speech Speed Scores
 * Recalculates speech_speed_score for sessions where it's 0 but WPM exists
 */

require('module-alias/register');
const database = require('@database');

/**
 * Value Range class from score.calculation.util.js
 */
class ValueRange {
  constructor(outerMin, bestMin, bestMax, outerMax) {
    this.outerMin = outerMin;
    this.bestMin = bestMin;
    this.bestMax = bestMax;
    this.outerMax = outerMax;
  }
}

/**
 * Linear equation: y = mx + t
 */
function linear(x, m, t) {
  return m * x + t;
}

/**
 * Maps an input value from one range to another linearly
 */
function linearRange(
  outputMin,
  outputMax,
  input,
  inputMin,
  inputMax,
  outputBorderedToMax = true,
  outputBorderedToMin = true
) {
  const inputRange = inputMax - inputMin;

  if (outputBorderedToMax && input > inputMax) {
    return outputMax;
  } else if (outputBorderedToMin && input < inputMin) {
    return outputMin;
  } else {
    const gradient = (outputMax - outputMin) / inputRange;
    return linear(input - inputMin, gradient, outputMin);
  }
}

/**
 * Calculates score based on value and range using trapezoidal algorithm
 *
 * Scoring Logic:
 * - value < outerMin → worst score (0)
 * - outerMin ≤ value < bestMin → linear increase 0→10
 * - bestMin ≤ value ≤ bestMax → best score (10)
 * - bestMax < value < outerMax → linear decrease 10→0
 * - value ≥ outerMax → worst score (0)
 */
function getScore(valueRange, value) {
  const WORST_SCORE = 0;
  const BEST_SCORE = 10;

  if (value < valueRange.outerMin) {
    return WORST_SCORE;
  } else if (value < valueRange.bestMin) {
    return linearRange(
      WORST_SCORE,
      BEST_SCORE,
      value,
      valueRange.outerMin,
      valueRange.bestMin
    );
  } else if (value <= valueRange.bestMax) {
    return BEST_SCORE;
  } else if (value < valueRange.outerMax) {
    return linearRange(
      BEST_SCORE,
      WORST_SCORE,
      value,
      valueRange.bestMax,
      valueRange.outerMax
    );
  } else {
    return WORST_SCORE;
  }
}

/**
 * Calculate speech speed score based on WPM
 */
function calculateSpeechSpeedScore(wpm) {
  // Optimal range: 95-145 WPM
  // Too slow (<70) or too fast (>180) = 0
  // Extended upper limit from 160 to 180 to be more forgiving with fast speakers
  return getScore(
    new ValueRange(70, 95, 145, 180),
    wpm
  );
}

async function fixSpeechSpeedScores() {
  try {
    console.log('\n====================================================');
    console.log('🔧 FIXING SPEECH SPEED SCORES');
    console.log('====================================================\n');

    // Get sessions with 0 speech_speed_score but valid WPM
    const problematicSessions = await database('presentation_session')
      .select('id', 'session_name', 'speech_speed_score', 'words_per_minute')
      .where('speech_speed_score', 0)
      .whereNotNull('words_per_minute')
      .where('words_per_minute', '>', 0);

    console.log(`Found ${problematicSessions.length} sessions to fix\n`);

    let fixed = 0;
    let skipped = 0;

    for (const session of problematicSessions) {
      const wpm = session.words_per_minute;

      // Skip obviously invalid WPM values
      if (wpm > 300) {
        console.log(`⚠️ Skipping ID ${session.id}: WPM ${wpm} is unrealistically high`);
        skipped++;
        continue;
      }

      const newScore = calculateSpeechSpeedScore(wpm);
      const roundedScore = Math.round(newScore * 100) / 100;

      // Update the database
      await database('presentation_session')
        .where('id', session.id)
        .update({ speech_speed_score: roundedScore });

      console.log(`✅ Fixed ID ${session.id}: ${session.session_name.substring(0, 30)}...`);
      console.log(`   WPM: ${wpm} → Score: ${roundedScore}`);

      fixed++;
    }

    console.log('\n====================================================');
    console.log(`✅ COMPLETE: Fixed ${fixed} sessions, skipped ${skipped}`);
    console.log('====================================================\n');

    // Verify the fix
    console.log('VERIFICATION - Sample of fixed sessions:\n');
    const verifiedSessions = await database('presentation_session')
      .select('id', 'session_name', 'speech_speed_score', 'words_per_minute')
      .whereIn('id', problematicSessions.slice(0, 5).map(s => s.id));

    verifiedSessions.forEach(s => {
      console.log(`ID ${s.id}: WPM=${s.words_per_minute}, Score=${s.speech_speed_score}`);
    });

    await database.destroy();
  } catch (error) {
    console.error('❌ Error:', error);
    await database.destroy();
  }
}

// Prompt for confirmation
const readline = require('readline');
const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout
});

console.log('\n⚠️ WARNING: This will update speech_speed_score for all sessions with 0 score but valid WPM.');
console.log('This will modify the database directly.\n');

rl.question('Do you want to proceed? (yes/no): ', (answer) => {
  if (answer.toLowerCase() === 'yes' || answer.toLowerCase() === 'y') {
    rl.close();
    fixSpeechSpeedScores();
  } else {
    console.log('Operation cancelled.');
    rl.close();
    process.exit(0);
  }
});