/**
 * Test API Response for Session
 * Check what data the API returns for a session
 */

require('module-alias/register');
const database = require('@database');

async function testApiResponse() {
  try {
    console.log('\n====================================================');
    console.log('🔍 TESTING API RESPONSE FOR SESSION 358');
    console.log('====================================================\n');

    // First, get the session directly from DB
    const session = await database('presentation_session')
      .where({ id: 358 })
      .first();

    console.log('📊 DATABASE VALUES:');
    console.log('  session_name:', session.session_name);
    console.log('  speech_speed_score:', session.speech_speed_score);
    console.log('  overall_score:', session.overall_score);
    console.log('  pauses_score:', session.pauses_score);
    console.log('  words_per_minute:', session.words_per_minute);

    // Now simulate what the API does
    console.log('\n🔄 SIMULATING API TRANSFORMATION:');

    // Get video blob (like API does)
    const videoBlob = await database('presentation_video_blobs')
      .where({ session_id: 358 })
      .first();

    // Parse JSON fields
    const parseJSON = (jsonString) => {
      if (!jsonString) return null;
      try {
        return JSON.parse(jsonString);
      } catch (e) {
        return null;
      }
    };

    // Build response like the API does
    const sessionData = {
      ...session,
      video_blob_key: videoBlob ? videoBlob.video_blob_key : null,
      transcript_words: parseJSON(session.transcript_words),
      transcript_sentences: parseJSON(session.transcript_sentences),
      transcribed_segments: parseJSON(session.transcribed_segments),
      filler_words: parseJSON(session.filler_words),
      key_phrases: parseJSON(session.key_phrases),
      key_words: parseJSON(session.key_words),
      audio_pitch_data: parseJSON(session.audio_pitch_data),
      audio_volume_data: parseJSON(session.audio_volume_data),
      audio_pauses_data: parseJSON(session.audio_pauses_data),
    };

    // Remove ugid (like API does)
    delete sessionData.ugid;

    console.log('\n📤 API RESPONSE DATA:');
    console.log('  speech_speed_score:', sessionData.speech_speed_score);
    console.log('  overall_score:', sessionData.overall_score);
    console.log('  pauses_score:', sessionData.pauses_score);
    console.log('  words_per_minute:', sessionData.words_per_minute);

    console.log('\n🔍 TYPE CHECK:');
    console.log('  Type of speech_speed_score:', typeof sessionData.speech_speed_score);
    console.log('  Is null?:', sessionData.speech_speed_score === null);
    console.log('  Is undefined?:', sessionData.speech_speed_score === undefined);
    console.log('  Is 0?:', sessionData.speech_speed_score === 0);

    console.log('\n====================================================');
    console.log('✅ TEST COMPLETE');
    console.log('====================================================\n');

    await database.destroy();
  } catch (error) {
    console.error('❌ Error:', error);
    await database.destroy();
  }
}

testApiResponse();