/**
 * Test Script for Azure Text Analytics Service
 *
 * This script tests the Text Analytics implementation with both
 * German and English text samples to ensure proper functionality.
 *
 * Usage: node src/test/testTextAnalytics.js
 */

// Set environment to development
process.env.NODE_ENV = 'development';

const textAnalyticsService = require('../services/textAnalytics.service');

// Test data
const germanText = `Das war eine großartige Präsentation. Die Folien waren sehr gut strukturiert und die Inhalte klar verständlich. Allerdings war das Tempo etwas zu schnell und einige wichtige Punkte wurden nur oberflächlich behandelt. Insgesamt aber eine sehr positive Erfahrung mit vielen wertvollen Informationen.`;

const englishText = `This was an excellent presentation. The slides were well-organized and the content was clearly explained. However, the pace was a bit too fast and some important points were only covered superficially. Overall, it was a very positive experience with lots of valuable information.`;

// Mock transcribed segments for sentence mapping
const mockSegments = [
  {
    text: "Das war eine großartige Präsentation.",
    offset: 0,
    duration: 3000,
    confidence: 0.95
  },
  {
    text: "Die Folien waren sehr gut strukturiert und die Inhalte klar verständlich.",
    offset: 3000,
    duration: 5000,
    confidence: 0.92
  },
  {
    text: "Allerdings war das Tempo etwas zu schnell und einige wichtige Punkte wurden nur oberflächlich behandelt.",
    offset: 8000,
    duration: 6000,
    confidence: 0.88
  },
  {
    text: "Insgesamt aber eine sehr positive Erfahrung mit vielen wertvollen Informationen.",
    offset: 14000,
    duration: 4000,
    confidence: 0.93
  }
];

async function testGermanAnalysis() {
  console.log('\n========================================');
  console.log('🇩🇪 TESTING GERMAN TEXT ANALYSIS');
  console.log('========================================\n');

  try {
    // Test 1: Sentiment Analysis
    console.log('1️⃣ Testing Sentiment Analysis...');
    const sentiment = await textAnalyticsService.analyzeSentiment(germanText, 'de');
    console.log('Sentiment Results:');
    console.log(`  ✅ Positive: ${(sentiment.positiveScore * 100).toFixed(2)}%`);
    console.log(`  ⚠️ Negative: ${(sentiment.negativeScore * 100).toFixed(2)}%`);
    console.log(`  ➖ Neutral: ${(sentiment.neutralScore * 100).toFixed(2)}%`);
    console.log();

    // Test 2: Key Phrase Extraction
    console.log('2️⃣ Testing Key Phrase Extraction...');
    const keyPhrases = await textAnalyticsService.extractKeyPhrases(germanText, 'de');
    console.log('Key Phrases:');
    keyPhrases.forEach((phrase, index) => {
      console.log(`  ${index + 1}. ${phrase}`);
    });
    console.log();

    // Test 3: Sentence-Level Analysis
    console.log('3️⃣ Testing Sentence-Level Sentiment Analysis...');
    const sentences = await textAnalyticsService.analyzeSentences(germanText, mockSegments, 'de');
    console.log('Analyzed Sentences:');
    sentences.forEach((sentence, index) => {
      console.log(`  Sentence ${index + 1}:`);
      console.log(`    Text: "${sentence.text.substring(0, 50)}..."`);
      console.log(`    Sentiment: ${sentence.sentiment}`);
      console.log(`    Scores: Pos=${(sentence.positiveScore * 100).toFixed(1)}% Neg=${(sentence.negativeScore * 100).toFixed(1)}% Neu=${(sentence.neutralScore * 100).toFixed(1)}%`);
      console.log(`    Timing: offset=${sentence.offset}ms, duration=${sentence.duration}ms`);
    });

  } catch (error) {
    console.error('❌ German analysis failed:', error.message);
  }
}

async function testEnglishAnalysis() {
  console.log('\n========================================');
  console.log('🇬🇧 TESTING ENGLISH TEXT ANALYSIS');
  console.log('========================================\n');

  try {
    // Test 1: Sentiment Analysis
    console.log('1️⃣ Testing Sentiment Analysis...');
    const sentiment = await textAnalyticsService.analyzeSentiment(englishText, 'en');
    console.log('Sentiment Results:');
    console.log(`  ✅ Positive: ${(sentiment.positiveScore * 100).toFixed(2)}%`);
    console.log(`  ⚠️ Negative: ${(sentiment.negativeScore * 100).toFixed(2)}%`);
    console.log(`  ➖ Neutral: ${(sentiment.neutralScore * 100).toFixed(2)}%`);
    console.log();

    // Test 2: Key Phrase Extraction
    console.log('2️⃣ Testing Key Phrase Extraction...');
    const keyPhrases = await textAnalyticsService.extractKeyPhrases(englishText, 'en');
    console.log('Key Phrases:');
    keyPhrases.forEach((phrase, index) => {
      console.log(`  ${index + 1}. ${phrase}`);
    });
    console.log();

    // Test 3: Sentence-Level Analysis
    console.log('3️⃣ Testing Sentence-Level Sentiment Analysis...');
    const sentences = await textAnalyticsService.analyzeSentences(englishText, [], 'en');
    console.log('Analyzed Sentences:');
    sentences.forEach((sentence, index) => {
      console.log(`  Sentence ${index + 1}:`);
      console.log(`    Text: "${sentence.text.substring(0, 50)}..."`);
      console.log(`    Sentiment: ${sentence.sentiment}`);
      console.log(`    Scores: Pos=${(sentence.positiveScore * 100).toFixed(1)}% Neg=${(sentence.negativeScore * 100).toFixed(1)}% Neu=${(sentence.neutralScore * 100).toFixed(1)}%`);
    });

  } catch (error) {
    console.error('❌ English analysis failed:', error.message);
  }
}

async function testLongText() {
  console.log('\n========================================');
  console.log('📚 TESTING LONG TEXT (CHUNKING)');
  console.log('========================================\n');

  // Create a very long text (over 5100 characters)
  const longText = germanText.repeat(50); // About 10,000 characters
  console.log(`Text length: ${longText.length} characters`);

  try {
    console.log('Testing chunking and analysis...');
    const sentiment = await textAnalyticsService.analyzeSentiment(longText, 'de');
    console.log('✅ Long text analysis successful!');
    console.log('Sentiment Results:');
    console.log(`  Positive: ${(sentiment.positiveScore * 100).toFixed(2)}%`);
    console.log(`  Negative: ${(sentiment.negativeScore * 100).toFixed(2)}%`);
    console.log(`  Neutral: ${(sentiment.neutralScore * 100).toFixed(2)}%`);

  } catch (error) {
    console.error('❌ Long text analysis failed:', error.message);
  }
}

async function testErrorHandling() {
  console.log('\n========================================');
  console.log('🔥 TESTING ERROR HANDLING');
  console.log('========================================\n');

  // Test with empty text
  console.log('1️⃣ Testing with empty text...');
  const emptyResult = await textAnalyticsService.analyzeSentiment('', 'de');
  console.log('Empty text result:', emptyResult);

  // Test with null text
  console.log('\n2️⃣ Testing with null text...');
  const nullResult = await textAnalyticsService.analyzeSentiment(null, 'de');
  console.log('Null text result:', nullResult);

  // Test with invalid language
  console.log('\n3️⃣ Testing with invalid language code...');
  try {
    const invalidLangResult = await textAnalyticsService.analyzeSentiment('Test text', 'xx');
    console.log('Invalid language result:', invalidLangResult);
  } catch (error) {
    console.log('Expected error for invalid language:', error.message);
  }
}

// Run all tests
async function runAllTests() {
  console.log('====================================================');
  console.log('🚀 AZURE TEXT ANALYTICS SERVICE TEST SUITE');
  console.log('====================================================');
  console.log(`📅 Date: ${new Date().toISOString()}`);
  console.log(`🔧 Environment: ${process.env.NODE_ENV}`);

  await testGermanAnalysis();
  await testEnglishAnalysis();
  await testLongText();
  await testErrorHandling();

  console.log('\n====================================================');
  console.log('✅ ALL TESTS COMPLETED');
  console.log('====================================================\n');
}

// Execute tests
runAllTests().catch(error => {
  console.error('Fatal error during testing:', error);
  process.exit(1);
});