/**
 * Update test session with non-zero pitch and volume data
 * For testing chart display
 */

require('module-alias/register');
const database = require('@database');

async function updateTestData(sessionId) {
  try {
    console.log('\n🔧 Updating session', sessionId, 'with test data...\n');

    // Generate realistic pitch data (human voice range: 80-400 Hz)
    const pitchData = [];
    const duration = 10000; // 10 seconds
    const interval = 100; // every 100ms

    for (let i = 0; i < duration; i += interval) {
      // Simulate natural voice pitch variation
      const basePitch = 200; // Average human voice
      const variation = Math.sin(i / 1000) * 50 + Math.random() * 30;
      pitchData.push({
        timestamp: i,
        value: Math.round(basePitch + variation)
      });
    }

    // Generate realistic volume data (0-100%)
    const volumeData = [];
    for (let i = 0; i < duration; i += interval) {
      // Simulate natural volume variation
      const baseVolume = 65;
      const variation = Math.sin(i / 500) * 20 + Math.random() * 10;
      volumeData.push({
        timestamp: i,
        value: Math.max(10, Math.min(100, Math.round(baseVolume + variation)))
      });
    }

    // Calculate averages
    const avgPitch = Math.round(pitchData.reduce((acc, p) => acc + p.value, 0) / pitchData.length);
    const avgVolume = Math.round(volumeData.reduce((acc, v) => acc + v.value, 0) / volumeData.length);

    console.log('📊 Generated Data:');
    console.log('   Pitch samples:', pitchData.length);
    console.log('   Volume samples:', volumeData.length);
    console.log('   Average pitch:', avgPitch, 'Hz');
    console.log('   Average volume:', avgVolume, '%');
    console.log('\nFirst 3 pitch samples:', pitchData.slice(0, 3));
    console.log('First 3 volume samples:', volumeData.slice(0, 3));

    // Update database
    const result = await database('presentation_session')
      .where('id', sessionId)
      .update({
        audio_pitch_data: JSON.stringify(pitchData),
        audio_volume_data: JSON.stringify(volumeData),
        avg_pitch: avgPitch,
        avg_volume: avgVolume
      });

    if (result) {
      console.log('\n✅ Session updated successfully!');
      console.log('Now check the charts in the frontend for session ID:', sessionId);
    } else {
      console.log('\n❌ Session not found or update failed');
    }

    await database.destroy();
  } catch (error) {
    console.error('❌ Error:', error);
    await database.destroy();
  }
}

// Get session ID from command line or use default
const sessionId = process.argv[2] || 360;
console.log('Usage: node src/test/updateTestData.js [sessionId]');
updateTestData(sessionId);