/**
 * Verify Pause Data in Database
 *
 * This script checks if pause analysis data is properly saved
 * and can be retrieved from the database.
 *
 * Usage: node src/test/verifyPauseData.js [sessionId]
 */

// Set environment to development
process.env.NODE_ENV = 'development';

// Setup module aliases
require('module-alias/register');

const database = require('@database');

async function verifyPauseData(sessionId) {
  try {
    console.log('\n====================================================');
    console.log('🔍 PAUSE DATA VERIFICATION');
    console.log('====================================================\n');

    // Fetch session from database
    const session = await database('presentation_session')
      .where({ id: sessionId })
      .first();

    if (!session) {
      console.error(`❌ Session with ID ${sessionId} not found`);
      return;
    }

    console.log(`✅ Session found: ${session.session_name}`);
    console.log(`📅 Created: ${session.recording_date}`);
    console.log('\n----------------------------------------');

    // Check Pause Data Field
    console.log('\n⏸️ PAUSE DATA ANALYSIS:');
    console.log(`Raw audio_pauses_data field: ${session.audio_pauses_data ? 'Present' : 'NULL/Empty'}`);

    if (session.audio_pauses_data) {
      try {
        const pauseData = JSON.parse(session.audio_pauses_data);
        console.log(`✅ Valid JSON - ${pauseData.length} pauses detected`);

        if (pauseData.length > 0) {
          console.log('\nFirst 5 pauses:');
          pauseData.slice(0, 5).forEach((pause, i) => {
            console.log(`  ${i + 1}. Offset: ${pause.offset || pause.timestamp}ms, Duration: ${pause.duration || pause.value}ms`);
          });

          // Calculate statistics
          const totalPauseTime = pauseData.reduce((sum, p) => sum + (p.duration || p.value || 0), 0);
          const avgPauseTime = pauseData.length > 0 ? totalPauseTime / pauseData.length : 0;

          console.log('\n📊 Pause Statistics:');
          console.log(`  Total Pauses: ${pauseData.length}`);
          console.log(`  Total Pause Time: ${(totalPauseTime / 1000).toFixed(2)}s`);
          console.log(`  Average Pause Duration: ${(avgPauseTime / 1000).toFixed(2)}s`);
        }
      } catch (e) {
        console.log('  ⚠️ Invalid JSON format:', e.message);
      }
    } else {
      console.log('  ❌ No pause data found (field is NULL or empty)');
    }

    // Check Related Pause Metrics
    console.log('\n📈 PAUSE METRICS:');
    console.log(`  Pauses per Minute: ${session.pauses_per_minute || 'Not calculated'}`);
    console.log(`  Average Pause Time: ${session.avg_pause_time ? session.avg_pause_time + 's' : 'Not calculated'}`);
    console.log(`  Total Pause Time: ${session.total_pause_time ? session.total_pause_time + 's' : 'Not calculated'}`);
    console.log(`  Pauses Score: ${session.pauses_score || 'Not calculated'}`);

    // Check if other audio data exists
    console.log('\n🎵 OTHER AUDIO DATA:');
    console.log(`  Pitch Data: ${session.audio_pitch_data ? 'Present' : 'Missing'}`);
    console.log(`  Volume Data: ${session.audio_volume_data ? 'Present' : 'Missing'}`);

    // Check key_words field (our newly implemented feature)
    console.log('\n🔑 KEY WORDS DATA:');
    if (session.key_words) {
      try {
        const keyWords = JSON.parse(session.key_words);
        console.log(`  Found ${keyWords.length} key words`);
        if (keyWords.length > 0) {
          console.log('  First 10 key words:', keyWords.slice(0, 10).join(', '));
        }
      } catch (e) {
        console.log('  ⚠️ Invalid JSON format');
      }
    } else {
      console.log('  ❌ No key words found');
    }

    console.log('\n====================================================');
    console.log('✅ VERIFICATION COMPLETE');
    console.log('====================================================\n');

  } catch (error) {
    console.error('❌ Error verifying pause data:', error);
  } finally {
    // Close database connection
    await database.destroy();
  }
}

async function listRecentSessionsWithPauses() {
  try {
    console.log('\n📋 SESSIONS WITH PAUSE DATA:');
    console.log('----------------------------------------');

    // Find sessions that have pause data
    const sessions = await database('presentation_session')
      .select('id', 'session_name', 'recording_date', 'pauses_per_minute', 'audio_pauses_data')
      .whereNotNull('audio_pauses_data')
      .orderBy('recording_date', 'desc')
      .limit(10);

    if (sessions.length === 0) {
      console.log('No sessions with pause data found in database');

      // List recent sessions anyway
      console.log('\n📋 RECENT SESSIONS (without pause data):');
      const recentSessions = await database('presentation_session')
        .select('id', 'session_name', 'recording_date')
        .orderBy('recording_date', 'desc')
        .limit(5);

      recentSessions.forEach(session => {
        const date = new Date(session.recording_date).toLocaleString();
        console.log(`  ID: ${session.id} | ${session.session_name} | ${date}`);
      });

      console.log('\n⚠️ None of these sessions have pause data!');
      console.log('This might indicate that pause detection is not working.');
      return;
    }

    sessions.forEach(session => {
      const date = new Date(session.recording_date).toLocaleString();
      let pauseCount = 0;
      try {
        const pauses = JSON.parse(session.audio_pauses_data);
        pauseCount = pauses.length;
      } catch (e) {
        pauseCount = 'Invalid';
      }
      console.log(`  ID: ${session.id} | ${session.session_name} | ${date} | Pauses: ${pauseCount} | PPM: ${session.pauses_per_minute || 'N/A'}`);
    });

    console.log('\nUse: node src/test/verifyPauseData.js [sessionId]');
    if (sessions.length > 0) {
      console.log('Example: node src/test/verifyPauseData.js ' + sessions[0].id);
    }

  } catch (error) {
    console.error('❌ Error listing sessions:', error);
  } finally {
    await database.destroy();
  }
}

// Main execution
const sessionId = process.argv[2];

if (sessionId) {
  verifyPauseData(parseInt(sessionId));
} else {
  listRecentSessionsWithPauses();
}