/**
 * Azure Storage Connection Test Script
 *
 * This script tests the Azure Blob Storage connection and creates required containers
 *
 * Container Structure:
 * - profile-images: User profile pictures
 * - presentation-materials: Presentation files organized by type
 *   - videos/: Video files (e.g., videos/presentation-123.mp4)
 *   - pdf/: PDF documents (e.g., pdf/slides-123.pdf)
 */

require("module-alias/register");
const azureStorage = require("./src/api/utils/azure.storage.util");

const REQUIRED_CONTAINERS = [
  "profile-images",
  "presentation-materials",
];

async function testAzureStorage() {
  console.log("=".repeat(60));
  console.log("🔵 AZURE BLOB STORAGE CONNECTION TEST");
  console.log("=".repeat(60));

  // Step 1: Test Connection
  console.log("\n📡 Step 1: Testing connection to Azure Blob Storage...");
  const connectionTest = await azureStorage.testConnection();

  if (!connectionTest.success) {
    console.error("❌ Connection test failed!");
    console.error(`Error: ${connectionTest.message}`);
    process.exit(1);
  }

  console.log("✅ Connection successful!");
  console.log(`Account Name: ${connectionTest.accountName}`);

  // Step 2: Create Required Containers
  console.log("\n📦 Step 2: Creating required containers...");
  for (const containerName of REQUIRED_CONTAINERS) {
    console.log(`\n  Creating container: ${containerName}`);
    const result = await azureStorage.createContainerIfNotExists(containerName);

    if (result.success) {
      if (result.exists) {
        console.log(`  ℹ️  ${result.message}`);
      } else {
        console.log(`  ✅ ${result.message}`);
      }
    } else {
      console.error(`  ❌ ${result.message}`);
    }
  }

  // Step 3: List All Containers
  console.log("\n📋 Step 3: Listing all containers...");
  const listResult = await azureStorage.listContainers();

  if (listResult.success) {
    console.log(`\n✅ Found ${listResult.containers.length} container(s):\n`);
    listResult.containers.forEach((container, index) => {
      console.log(`  ${index + 1}. ${container.name}`);
      console.log(`     Last Modified: ${container.lastModified}`);
      console.log(`     Public Access: ${container.publicAccess || "private"}\n`);
    });
  } else {
    console.error(`❌ ${listResult.message}`);
  }

  // Summary
  console.log("=".repeat(60));
  console.log("✅ Azure Blob Storage test completed successfully!");
  console.log("=".repeat(60));
}

// Run the test
testAzureStorage()
  .then(() => {
    console.log("\n✨ All operations completed successfully!");
    process.exit(0);
  })
  .catch((error) => {
    console.error("\n❌ Test failed with error:");
    console.error(error);
    process.exit(1);
  });
