/**
 * Get Session Controller
 *
 * Returns a single presentation session by ID
 * - Validates session exists
 * - Checks authorization (user can only view their own sessions)
 * - Returns complete session data including analysis and AI feedback
 */

const database = require("@database");
const utils = require("@utils");

const getSession = async (req, res) => {
  try {
    // Get user UGID from token
    const { ugid } = req.user;

    // Get session ID from params (already validated by Joi)
    const sessionId = req.params.id;

    // Fetch session from database
    const session = await database("presentation_session")
      .where({ id: sessionId })
      .first();

    // Check if session exists
    if (!session) {
      return utils.responseSnippet(res, 404, "Session not found", null);
    }

    // Authorization check: User can only view their own sessions
    if (session.ugid !== ugid) {
      return utils.responseSnippet(
        res,
        403,
        "You do not have permission to view this session",
        null
      );
    }

    // Fetch AI feedback if exists
    const aiFeedback = await database("presentation_feedback")
      .where({ session_id: sessionId })
      .orderBy("created_at", "desc")
      .first();

    // Fetch video blob info from the new table
    const videoBlob = await database("presentation_video_blobs")
      .where({ session_id: sessionId })
      .first();

    // Parse JSON fields
    const parseJSON = (jsonString) => {
      if (!jsonString) return null;
      try {
        return JSON.parse(jsonString);
      } catch (e) {
        return null;
      }
    };

    // Build response with parsed JSON fields
    const sessionData = {
      ...session,
      // Add video_blob_key from the new table
      video_blob_key: videoBlob ? videoBlob.video_blob_key : null,
      // Parse JSON columns
      // slides: parseJSON(session.slides), // REMOVED
      // slide_timestamps: parseJSON(session.slide_timestamps), // REMOVED
      // heatmap: parseJSON(session.heatmap), // REMOVED
      // averages: parseJSON(session.averages), // REMOVED
      transcript_words: parseJSON(session.transcript_words),
      transcript_sentences: parseJSON(session.transcript_sentences),
      transcribed_segments: parseJSON(session.transcribed_segments),
      filler_words: parseJSON(session.filler_words),
      key_phrases: parseJSON(session.key_phrases),
      key_words: parseJSON(session.key_words),
      audio_pitch_data: parseJSON(session.audio_pitch_data),
      audio_volume_data: parseJSON(session.audio_volume_data),
      audio_pauses_data: parseJSON(session.audio_pauses_data),
      // audio_timed_averages: parseJSON(session.audio_timed_averages), // REMOVED
      // Add AI feedback if available
      ai_feedback: aiFeedback
        ? {
            id: aiFeedback.id,
            feedback_text: aiFeedback.feedback_text,
            language: aiFeedback.language,
            model_version: aiFeedback.model_version,
            created_at: aiFeedback.created_at,
          }
        : null,
    };

    // Remove ugid from response
    delete sessionData.ugid;

    // Debug logging for audio data
    console.log('🎵 Backend Audio Data Response:', {
      sessionId: sessionId,
      hasPitchData: !!sessionData.audio_pitch_data,
      pitchDataLength: sessionData.audio_pitch_data ? sessionData.audio_pitch_data.length : 0,
      hasVolumeData: !!sessionData.audio_volume_data,
      volumeDataLength: sessionData.audio_volume_data ? sessionData.audio_volume_data.length : 0,
      samplePitch: sessionData.audio_pitch_data ? sessionData.audio_pitch_data.slice(0, 2) : null
    });

    // Debug logging for scores
    console.log('🎯 Backend Score Data:', {
      sessionId: sessionId,
      speech_speed_score: sessionData.speech_speed_score,
      overall_score: sessionData.overall_score,
      pauses_score: sessionData.pauses_score,
      repetition_score: sessionData.repetition_score,
      filler_words_score: sessionData.filler_words_score,
      speaking_clearly_score: sessionData.speaking_clearly_score
    });

    return utils.responseSnippet(
      res,
      200,
      "Session retrieved successfully",
      sessionData
    );
  } catch (error) {
    console.error("Get session error:", error);
    return utils.catchErrorHandlerSnippet(res, error, __dirname);
  }
};

module.exports = getSession;
