const authDocuments = {
  codegenerator: {
    post: {
      tags: ["Authentication"],
      summary: "Generate verification code",
      description: "Sends a verification code to the provided email address",
      requestBody: {
        required: true,
        content: {
          "application/json": {
            schema: {
              type: "object",
              required: ["email"],
              properties: {
                email: {
                  type: "string",
                  format: "email",
                  description: "The user's email address",
                  example: "user@example.com",
                },
              },
            },
          },
        },
      },
      responses: {
        200: {
          description: "Verification code sent successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  message: {
                    type: "string",
                    example: "Verification code has been sent to your email.",
                  },
                  data: {
                    type: "null",
                  },
                },
              },
            },
          },
        },
        429: {
          description: "Too many requests, rate limit exceeded",
        },
        403: {
          description: "Email not registered (in admin based mode)",
        },
        500: {
          description: "Internal server error",
        },
      },
    },
  },

  "verify/{code}/{email}": {
    get: {
      tags: ["Authentication"],
      summary: "Verify authentication code",
      description:
        "Verifies the provided code for the email address and returns an authentication token",
      parameters: [
        {
          in: "path",
          name: "code",
          schema: {
            type: "string",
          },
          required: true,
          description: "Verification code sent to the user's email",
        },
        {
          in: "path",
          name: "email",
          schema: {
            type: "string",
            format: "email",
          },
          required: true,
          description: "The user's email address",
        },
      ],
      responses: {
        200: {
          description: "Authentication successful",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  message: {
                    type: "string",
                    example: "Authentication successful",
                  },
                  data: {
                    type: "object",
                    properties: {
                      token: {
                        type: "string",
                        description: "JWT authentication token",
                      },
                      user: {
                        type: "object",
                        properties: {
                          email: {
                            type: "string",
                            format: "email",
                          },
                          ugid: {
                            type: "string",
                            format: "uuid",
                          },
                        },
                      },
                    },
                  },
                },
              },
            },
          },
        },
        400: {
          description: "Invalid verification code",
        },
        401: {
          description: "Too many failed attempts",
        },
        404: {
          description: "Verification code not found",
        },
        500: {
          description: "Internal server error",
        },
      },
    },
  },

  info: {
    get: {
      tags: ["Authentication"],
      summary: "Get user information",
      description: "Retrieves authenticated user information",
      security: [
        {
          bearerAuth: [],
        },
      ],
      responses: {
        200: {
          description: "User information retrieved successfully",
          content: {
            "application/json": {
              schema: {
                type: "object",
                properties: {
                  message: {
                    type: "string",
                    example: "User information retrieved successfully.",
                  },
                  data: {
                    type: "object",
                    properties: {
                      ugid: {
                        type: "string",
                        format: "uuid",
                      },
                      username: {
                        type: "string",
                      },
                      name: {
                        type: "string",
                      },
                      email: {
                        type: "string",
                        format: "email",
                      },
                      region: {
                        type: "string",
                      },
                      timeJoined: {
                        type: "string",
                        format: "date-time",
                      },
                      lastIP: {
                        type: "string",
                      },
                      accessLevel: {
                        type: "string",
                        nullable: true,
                      },
                      devices: {
                        type: "array",
                        items: {
                          type: "object",
                          properties: {
                            device_id: {
                              type: "string",
                            },
                            device_name: {
                              type: "string",
                            },
                            device_type: {
                              type: "string",
                            },
                            last_active: {
                              type: "string",
                              format: "date-time",
                            },
                            is_online: {
                              type: "boolean",
                            },
                            created_at: {
                              type: "string",
                              format: "date-time",
                            },
                          },
                        },
                      },
                    },
                  },
                },
              },
            },
          },
        },
        401: {
          description: "Authentication required or invalid token",
        },
        404: {
          description: "User not found",
        },
        500: {
          description: "Internal server error",
        },
      },
    },
  },
};

module.exports = authDocuments;
