/**
 * OpenAI Service for AI Feedback Generation
 *
 * This service integrates with OpenAI API to generate structured feedback
 * for presentation sessions based on analysis data.
 */

const axios = require('axios');
const config = require('@config');

class OpenAIService {
  constructor() {
    this.apiKey = config.openai?.apiKey || process.env.OPENAI_API_KEY;
    this.apiUrl = 'https://api.openai.com/v1/chat/completions';
    this.model = config.openai?.model || 'gpt-4';
    this.temperature = config.openai?.temperature || 0.7;
    this.maxTokens = config.openai?.maxTokens || 2500;
  }

  /**
   * Generate structured feedback for a presentation session
   * @param {Object} analysisData - Session analysis data
   * @param {string} language - Target language ('de' or 'en')
   * @returns {Promise<string>} JSON string containing structured feedback
   */
  async generateStructuredFeedbackAsync(analysisData, language = 'de') {
    if (!this.apiKey) {
      throw new Error('OpenAI API key is not configured');
    }

    const prompt = this.buildFeedbackPrompt(analysisData, language);

    try {
      const response = await axios.post(
        this.apiUrl,
        {
          model: this.model,
          messages: [
            {
              role: 'system',
              content: 'You are an expert presentation coach providing constructive feedback. You MUST respond with valid JSON only. Do not include any markdown formatting or code blocks. Return only the JSON object.'
            },
            {
              role: 'user',
              content: prompt
            }
          ],
          temperature: this.temperature,
          max_tokens: this.maxTokens
        },
        {
          headers: {
            'Authorization': `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return response.data.choices[0].message.content;
    } catch (error) {
      console.error('OpenAI API Error:', error.response?.data || error.message);
      throw new Error('Failed to generate AI feedback');
    }
  }

  /**
   * Build the feedback prompt with session data
   * @param {Object} context - Session analysis context
   * @param {string} language - Target language
   * @returns {string} Formatted prompt for OpenAI
   */
  buildFeedbackPrompt(context, language) {
    const fillerWordsText = context.fillerWords?.length > 0
      ? context.fillerWords.slice(0, 5).map(f => f.word).join(', ')
      : 'none';

    const targetLanguage = language === 'de' ? 'German (Deutsch)' : 'English';
    const languageExample = language === 'de'
      ? 'Example in German: "Sie haben eine sehr gute Präsentation gehalten. Ihre Sprechgeschwindigkeit war optimal."'
      : 'Example in English: "You delivered a very good presentation. Your speaking pace was optimal."';

    return `=== CRITICAL INSTRUCTIONS - READ CAREFULLY ===

LANGUAGE REQUIREMENT #1: You MUST write ALL content in ${targetLanguage}.
LANGUAGE REQUIREMENT #2: Every single word in your response must be in ${targetLanguage}.
LANGUAGE REQUIREMENT #3: The summary, strengths, improvements, recommendations, and closing message MUST ALL be in ${targetLanguage}.
LANGUAGE REQUIREMENT #4: Do NOT mix languages. Do NOT use English if the language is German. Do NOT use German if the language is English.
LANGUAGE REQUIREMENT #5: ${languageExample}

OUTPUT FORMAT REQUIREMENT #1: You MUST return ONLY valid JSON. No markdown. No code blocks. No explanations.
OUTPUT FORMAT REQUIREMENT #2: Do NOT wrap your response in \`\`\`json or \`\`\` markers.
OUTPUT FORMAT REQUIREMENT #3: Do NOT add any text before or after the JSON object.
OUTPUT FORMAT REQUIREMENT #4: Your response must start with { and end with }.
OUTPUT FORMAT REQUIREMENT #5: Return PURE JSON only. Nothing else.

=== PRESENTATION SESSION DATA ===

SESSION OVERVIEW:
- Duration: ${context.sessionLength || 0} minutes
- Overall Performance Score: ${context.overallScore || 0}/100

SPEECH DELIVERY METRICS:
- Words per Minute (WPM): ${context.wordsPerMinute || 0} (optimal range: 140-160 WPM)
- Speech Speed Score: ${context.speechSpeedScore || 0}/100
- Speaking Clarity Score: ${context.speakingClearlyScore || 0}/100

PAUSE PATTERN ANALYSIS:
- Pauses per Minute: ${context.pausesPerMinute || 0}
- Total Pause Duration: ${context.totalPauseTime || 0} seconds
- Pause Quality Score: ${context.pausesScore || 0}/100

VOCAL CHARACTERISTICS:
- Average Pitch: ${context.averagePitch || 0} Hz
- Average Volume Level: ${context.averageVolume || 0} dB

CONTENT QUALITY METRICS:
- Sentiment Score: ${context.sentiment || 0} (scale: 0=negative, 1=positive)
- Filler Words Count: ${context.fillerWordCount || 0} occurrences (examples: ${fillerWordsText})
- Filler Words Score: ${context.fillerWordsScore || 0}/100
- Repetition Count: ${context.repetitionCount || 0}
- Repetition Score: ${context.repetitionScore || 0}/100

FULL PRESENTATION TRANSCRIPTION:
${context.transcriptionText || ''}

=== YOUR TASK ===

Analyze the above presentation data and provide constructive feedback in the following JSON structure.

You must provide comprehensive analysis in these areas:
1. Technical delivery metrics (speech speed, pauses, filler words)
2. Presentation structure evaluation (introduction, body, conclusion, logical flow)
3. Argumentation plausibility (claims, evidence, reasoning coherence)

REMEMBER: All text content MUST be in ${targetLanguage}.

JSON STRUCTURE - FOLLOW THIS EXACTLY:

{
  "summary": "Write a 2-3 sentence overall assessment in ${targetLanguage}. Mention the presenter's overall performance and key takeaway.",

  "strengths": [
    "Write first strength in ${targetLanguage}",
    "Write second strength in ${targetLanguage}",
    "Write third strength in ${targetLanguage}"
  ],

  "areasForImprovement": [
    "Write first area for improvement in ${targetLanguage}",
    "Write second area for improvement in ${targetLanguage}",
    "Write third area for improvement in ${targetLanguage}"
  ],

  "presentationStructure": "Analyze and evaluate the structure of the presentation in ${targetLanguage}. Use specific elements to explain your evaluation in a compact format (approx. half a DIN A4 page). If necessary, provide concrete improvement suggestions and explain how these could optimize the structure. Focus on: introduction clarity, body organization, conclusion effectiveness, logical flow, and transitions between topics.",

  "argumentationPlausibility": "Evaluate the plausibility of the argumentation in the presentation in ${targetLanguage}. Address the arguments briefly, identify strengths and weaknesses, and explain precisely which aspects could be improved. Offer specific suggestions to increase credibility. This should be compact (approx. half a DIN A4 page). Focus on: claim clarity, evidence quality, reasoning logic, coherence, and persuasiveness.",

  "recommendations": [
    {
      "category": "${language === 'de' ? 'Sprechgeschwindigkeit' : 'Speech Speed'}",
      "observation": "Describe what you observed about speech speed in ${targetLanguage}. Compare ${context.wordsPerMinute || 0} WPM to optimal 140-160 WPM range.",
      "suggestion": "Provide specific actionable advice in ${targetLanguage} on how to improve speech speed.",
      "metrics": {
        "currentValue": ${context.wordsPerMinute || 0},
        "optimalValue": 150.0,
        "unit": "WPM"
      }
    },
    {
      "category": "${language === 'de' ? 'Pausen' : 'Pauses'}",
      "observation": "Describe what you observed about pause patterns in ${targetLanguage}. Mention ${context.pausesPerMinute || 0} pauses per minute.",
      "suggestion": "Provide specific actionable advice in ${targetLanguage} on how to use pauses more effectively.",
      "metrics": {
        "currentValue": ${context.pausesPerMinute || 0},
        "optimalValue": 2.0,
        "unit": "per minute"
      }
    },
    {
      "category": "${language === 'de' ? 'Füllwörter' : 'Filler Words'}",
      "observation": "Describe what you observed about filler word usage in ${targetLanguage}. Mention ${context.fillerWordCount || 0} occurrences.",
      "suggestion": "Provide specific actionable advice in ${targetLanguage} on how to reduce filler words.",
      "metrics": {
        "currentValue": ${context.fillerWordCount || 0},
        "optimalValue": 0,
        "unit": "occurrences"
      }
    }
  ],

  "closingMessage": "Write an encouraging closing message in ${targetLanguage}. Be supportive and motivating. End with good wishes for future presentations."
}`;
  }

  /**
   * Extract analysis data from session for feedback generation
   * @param {Object} session - Session data from database
   * @returns {Object} Formatted analysis data
   */
  extractAnalysisData(session) {
    // Calculate filler words count
    let fillerWordCount = 0;
    let fillerWords = [];
    try {
      if (session.filler_words) {
        fillerWords = JSON.parse(session.filler_words);
        fillerWordCount = fillerWords.length;
      }
    } catch (e) {
      console.warn('Failed to parse filler_words:', e);
    }

    // Calculate sentiment
    const sentiment = session.sentiment_positive_score || 0;

    return {
      sessionLength: Math.round(session.session_length / 60), // Convert to minutes
      overallScore: session.overall_score || 0,
      wordsPerMinute: session.words_per_minute || 0,
      speechSpeedScore: session.speech_speed_score || 0,
      speakingClearlyScore: session.speaking_clearly_score || 0,
      pausesPerMinute: session.pauses_per_minute || 0,
      totalPauseTime: session.total_pause_time || 0,
      pausesScore: session.pauses_score || 0,
      averagePitch: session.avg_pitch || 0,
      averageVolume: session.avg_volume || 0,
      sentiment: sentiment / 100, // Convert to 0-1 scale
      fillerWordCount: fillerWordCount,
      fillerWords: fillerWords,
      fillerWordsScore: session.filler_words_score || 0,
      repetitionCount: session.repetition_count || 0,
      repetitionScore: session.repetition_score || 0,
      transcriptionText: session.transcript_text || ''
    };
  }
}

module.exports = new OpenAIService();