const { BlobServiceClient } = require("@azure/storage-blob");
const config = require("@config");

/**
 * Azure Blob Storage Utility
 * Provides functions for interacting with Azure Blob Storage
 */

let blobServiceClient = null;

/**
 * Initialize Blob Service Client
 * @returns {BlobServiceClient}
 */
const getBlobServiceClient = () => {
  if (!blobServiceClient) {
    const connectionString = config.azureStorage.connectionString;

    if (!connectionString) {
      throw new Error("Azure Storage connection string not configured");
    }

    blobServiceClient = BlobServiceClient.fromConnectionString(connectionString);
  }

  return blobServiceClient;
};

/**
 * Test connection to Azure Blob Storage
 * @returns {Promise<{success: boolean, message: string, accountName?: string}>}
 */
const testConnection = async () => {
  try {
    const client = getBlobServiceClient();

    // Try to list containers to verify connection
    const iterator = client.listContainers().byPage({ maxPageSize: 1 });
    await iterator.next();

    return {
      success: true,
      message: "Successfully connected to Azure Blob Storage",
      accountName: config.azureStorage.accountName,
    };
  } catch (error) {
    console.error("Azure Storage connection test failed:", error);
    return {
      success: false,
      message: `Connection failed: ${error.message}`,
    };
  }
};

/**
 * Create a container if it doesn't exist
 * @param {string} containerName - Name of the container to create
 * @returns {Promise<{success: boolean, message: string, exists?: boolean}>}
 */
const createContainerIfNotExists = async (containerName) => {
  try {
    const client = getBlobServiceClient();
    const containerClient = client.getContainerClient(containerName);

    // Check if container exists
    const exists = await containerClient.exists();

    if (exists) {
      return {
        success: true,
        message: `Container '${containerName}' already exists`,
        exists: true,
      };
    }

    // Create container as private (no public access)
    const createResponse = await containerClient.create();

    return {
      success: true,
      message: `Container '${containerName}' created successfully`,
      exists: false,
      requestId: createResponse.requestId,
    };
  } catch (error) {
    console.error(`Error creating container '${containerName}':`, error);
    return {
      success: false,
      message: `Failed to create container: ${error.message}`,
    };
  }
};

/**
 * List all containers
 * @returns {Promise<{success: boolean, containers?: Array, message?: string}>}
 */
const listContainers = async () => {
  try {
    const client = getBlobServiceClient();
    const containers = [];

    for await (const container of client.listContainers()) {
      containers.push({
        name: container.name,
        lastModified: container.properties.lastModified,
        publicAccess: container.properties.publicAccess,
      });
    }

    return {
      success: true,
      containers,
    };
  } catch (error) {
    console.error("Error listing containers:", error);
    return {
      success: false,
      message: `Failed to list containers: ${error.message}`,
    };
  }
};

/**
 * Upload a file to Azure Blob Storage
 * @param {string} containerName - Container name
 * @param {string} blobName - Blob name (file name in storage)
 * @param {Buffer} fileBuffer - File buffer to upload
 * @param {Object} options - Upload options (contentType, metadata, etc.)
 * @returns {Promise<{success: boolean, url?: string, message?: string}>}
 */
const uploadFile = async (containerName, blobName, fileBuffer, options = {}) => {
  try {
    const client = getBlobServiceClient();
    const containerClient = client.getContainerClient(containerName);
    const blockBlobClient = containerClient.getBlockBlobClient(blobName);

    const uploadOptions = {
      blobHTTPHeaders: {
        blobContentType: options.contentType || "application/octet-stream",
      },
      metadata: options.metadata || {},
    };

    await blockBlobClient.upload(fileBuffer, fileBuffer.length, uploadOptions);

    return {
      success: true,
      url: blockBlobClient.url,
      message: `File '${blobName}' uploaded successfully to container '${containerName}'`,
    };
  } catch (error) {
    console.error(`Error uploading file to '${containerName}/${blobName}':`, error);
    return {
      success: false,
      message: `Failed to upload file: ${error.message}`,
    };
  }
};

/**
 * Delete a blob from Azure Blob Storage
 * @param {string} containerName - Container name
 * @param {string} blobName - Blob name to delete
 * @returns {Promise<{success: boolean, message: string}>}
 */
const deleteFile = async (containerName, blobName) => {
  try {
    const client = getBlobServiceClient();
    const containerClient = client.getContainerClient(containerName);
    const blockBlobClient = containerClient.getBlockBlobClient(blobName);

    await blockBlobClient.deleteIfExists();

    return {
      success: true,
      message: `File '${blobName}' deleted successfully from container '${containerName}'`,
    };
  } catch (error) {
    console.error(`Error deleting file '${containerName}/${blobName}':`, error);
    return {
      success: false,
      message: `Failed to delete file: ${error.message}`,
    };
  }
};

/**
 * Stream a file from Azure Blob Storage (proxy without downloading to memory)
 * @param {string} containerName - Container name
 * @param {string} blobName - Blob name to stream
 * @returns {Promise<{success: boolean, stream?: ReadableStream, contentType?: string, contentLength?: number, message?: string}>}
 */
const streamFile = async (containerName, blobName) => {
  try {
    const client = getBlobServiceClient();
    const containerClient = client.getContainerClient(containerName);
    const blockBlobClient = containerClient.getBlockBlobClient(blobName);

    // Check if blob exists
    const exists = await blockBlobClient.exists();
    if (!exists) {
      return {
        success: false,
        message: `File '${blobName}' not found in container '${containerName}'`,
      };
    }

    // Get blob properties to retrieve content type and length
    const properties = await blockBlobClient.getProperties();

    // Get blob as a readable stream (no download to memory!)
    const downloadResponse = await blockBlobClient.download(0);

    return {
      success: true,
      stream: downloadResponse.readableStreamBody,
      contentType: properties.contentType || "application/octet-stream",
      contentLength: properties.contentLength,
      metadata: properties.metadata,
    };
  } catch (error) {
    console.error(`Error streaming file '${containerName}/${blobName}':`, error);
    return {
      success: false,
      message: `Failed to stream file: ${error.message}`,
    };
  }
};

/**
 * Get a blob URL
 * @param {string} containerName - Container name
 * @param {string} blobName - Blob name
 * @returns {string} - Full URL to the blob
 */
const getBlobUrl = (containerName, blobName) => {
  const client = getBlobServiceClient();
  const containerClient = client.getContainerClient(containerName);
  const blockBlobClient = containerClient.getBlockBlobClient(blobName);
  return blockBlobClient.url;
};

module.exports = {
  testConnection,
  createContainerIfNotExists,
  listContainers,
  uploadFile,
  streamFile,
  downloadFile: streamFile, // Backward compatibility alias
  deleteFile,
  getBlobUrl,
  getBlobServiceClient,
};
