/**
 * Test Audio Analysis Service
 *
 * This script tests the audio analysis functionality
 * to ensure pitch and volume data are properly extracted.
 *
 * Usage: node src/test/testAudioAnalysis.js [sessionId]
 */

// Set environment to development
process.env.NODE_ENV = 'development';

// Setup module aliases
require('module-alias/register');

const database = require('@database');
const audioAnalysisService = require('../services/audioAnalysis.service');
const fs = require('fs').promises;
const path = require('path');

async function testAudioAnalysis(sessionId) {
  try {
    console.log('\n====================================================');
    console.log('🎵 AUDIO ANALYSIS TEST');
    console.log('====================================================\n');

    // Fetch session from database
    const session = await database('presentation_session')
      .where({ id: sessionId })
      .first();

    if (!session) {
      console.error(`❌ Session with ID ${sessionId} not found`);
      return;
    }

    console.log(`✅ Session found: ${session.session_name}`);
    console.log(`📅 Created: ${session.recording_date}`);
    console.log('\n----------------------------------------');

    // Check existing audio data
    console.log('\n📊 EXISTING AUDIO DATA:');
    console.log(`  Pitch Data: ${session.audio_pitch_data ? 'Present' : 'Missing'}`);
    console.log(`  Volume Data: ${session.audio_volume_data ? 'Present' : 'Missing'}`);
    console.log(`  Avg Pitch: ${session.avg_pitch || 'N/A'}`);
    console.log(`  Avg Volume: ${session.avg_volume || 'N/A'}`);

    if (session.audio_pitch_data) {
      try {
        const pitchData = JSON.parse(session.audio_pitch_data);
        console.log(`  Pitch samples: ${pitchData.length}`);
        if (pitchData.length > 0) {
          console.log(`  First pitch: ${pitchData[0].value} at ${pitchData[0].timestamp}ms`);
        }
      } catch (e) {
        console.log('  ⚠️ Invalid pitch data JSON');
      }
    }

    if (session.audio_volume_data) {
      try {
        const volumeData = JSON.parse(session.audio_volume_data);
        console.log(`  Volume samples: ${volumeData.length}`);
        if (volumeData.length > 0) {
          console.log(`  First volume: ${volumeData[0].value}% at ${volumeData[0].timestamp}ms`);
        }
      } catch (e) {
        console.log('  ⚠️ Invalid volume data JSON');
      }
    }

    console.log('\n====================================================');
    console.log('✅ AUDIO ANALYSIS TEST COMPLETE');
    console.log('====================================================\n');

  } catch (error) {
    console.error('❌ Error testing audio analysis:', error);
  } finally {
    // Close database connection
    await database.destroy();
  }
}

async function testServiceDirectly() {
  try {
    console.log('\n====================================================');
    console.log('🎵 TESTING AUDIO ANALYSIS SERVICE DIRECTLY');
    console.log('====================================================\n');

    // Create a test audio file (silent audio)
    const testVideoPath = path.join(process.cwd(), 'temp', 'test-video.mp4');

    // Check if FFmpeg is installed
    const { exec } = require('child_process');
    const util = require('util');
    const execPromise = util.promisify(exec);

    console.log('🔍 Checking FFmpeg installation...');
    try {
      const { stdout } = await execPromise('ffmpeg -version');
      console.log('✅ FFmpeg is installed');
      console.log(stdout.split('\n')[0]); // Print version
    } catch (error) {
      console.error('❌ FFmpeg is not installed!');
      console.error('Please install FFmpeg: brew install ffmpeg (Mac) or apt-get install ffmpeg (Linux)');
      return;
    }

    // Create a simple test video with audio
    console.log('\n🎬 Creating test video with audio...');
    await execPromise(`ffmpeg -y -f lavfi -i "sine=frequency=440:duration=5" -f lavfi -i "color=black:size=640x480:duration=5" -c:v libx264 -c:a aac "${testVideoPath}"`);

    console.log('✅ Test video created');

    // Test the audio analysis
    console.log('\n🎵 Analyzing audio...');
    const result = await audioAnalysisService.analyzeAudio(testVideoPath, 100);

    console.log('\n📊 ANALYSIS RESULTS:');
    console.log(`  Pitch samples: ${result.pitchData.length}`);
    console.log(`  Volume samples: ${result.volumeData.length}`);
    console.log(`  Average Pitch: ${result.avgPitch} Hz`);
    console.log(`  Average Volume: ${result.avgVolume}%`);

    if (result.pitchData.length > 0) {
      console.log(`  First 3 pitch values:`);
      result.pitchData.slice(0, 3).forEach(p => {
        console.log(`    ${p.timestamp}ms: ${p.value} Hz`);
      });
    }

    if (result.volumeData.length > 0) {
      console.log(`  First 3 volume values:`);
      result.volumeData.slice(0, 3).forEach(v => {
        console.log(`    ${v.timestamp}ms: ${v.value}%`);
      });
    }

    // Clean up
    await fs.unlink(testVideoPath);
    console.log('\n✅ Test video cleaned up');

  } catch (error) {
    console.error('❌ Service test failed:', error);
  }
}

async function listSessionsWithAudioData() {
  try {
    console.log('\n📋 SESSIONS WITH AUDIO DATA:');
    console.log('----------------------------------------');

    const sessions = await database('presentation_session')
      .select('id', 'session_name', 'recording_date', 'audio_pitch_data', 'audio_volume_data', 'avg_pitch', 'avg_volume')
      .whereNotNull('audio_pitch_data')
      .orWhereNotNull('audio_volume_data')
      .orderBy('recording_date', 'desc')
      .limit(10);

    if (sessions.length === 0) {
      console.log('No sessions with audio data found');

      console.log('\n📋 RECENT SESSIONS (without audio data):');
      const recentSessions = await database('presentation_session')
        .select('id', 'session_name', 'recording_date')
        .orderBy('recording_date', 'desc')
        .limit(5);

      recentSessions.forEach(session => {
        const date = new Date(session.recording_date).toLocaleString();
        console.log(`  ID: ${session.id} | ${session.session_name} | ${date}`);
      });

      console.log('\n⚠️ None of these sessions have audio analysis data!');
    } else {
      sessions.forEach(session => {
        const date = new Date(session.recording_date).toLocaleString();
        let pitchCount = 0;
        let volumeCount = 0;

        try {
          if (session.audio_pitch_data) {
            const pitch = JSON.parse(session.audio_pitch_data);
            pitchCount = pitch.length;
          }
        } catch (e) {}

        try {
          if (session.audio_volume_data) {
            const volume = JSON.parse(session.audio_volume_data);
            volumeCount = volume.length;
          }
        } catch (e) {}

        console.log(`  ID: ${session.id} | ${session.session_name} | Pitch: ${pitchCount} samples | Volume: ${volumeCount} samples`);
      });
    }

    console.log('\nUsage:');
    console.log('  node src/test/testAudioAnalysis.js [sessionId] - Check specific session');
    console.log('  node src/test/testAudioAnalysis.js test - Test service directly');

  } catch (error) {
    console.error('❌ Error listing sessions:', error);
  } finally {
    await database.destroy();
  }
}

// Main execution
const arg = process.argv[2];

if (arg === 'test') {
  testServiceDirectly();
} else if (arg) {
  testAudioAnalysis(parseInt(arg));
} else {
  listSessionsWithAudioData();
}